#! /bin/bash

set -e

#
# Script to update version information in header file fcc/fvw/FVSourceRevision.h and optionally in macOS/iOS info.plist files
#
# Xcode projects depending on FVSourceRevision.h be must run this script before compile phase.
# Xcode projects updating .plist files of privileged helper tools must run this script before linker
#

# Root directory of working copy of source tree
WORK_ROOT=$1

#Name of Info.plist file to update (optional)
PLIST_FILE=$2

USAGE_INFO="update_version_info.sh <Source Root Directory> [<Plist Filename>]\n"


if [ -z "${WORK_ROOT}"  ] ; then
echo "source root directory undefined"
echo ""
echo $USAGE_INFO
exit -1;
fi

if ! git diff-index --quiet HEAD --
then
    UNCOMMITTED_CHANGES="M"
else
    UNCOMMITTED_CHANGES=""
fi

if [ -z "${CI_PIPELINE_ID}" ]
then
#
# If GitLab Pipeline id is undefined, compute base for GIT_REVISION_CNT to make sure that revision count from
# production branch builds is higher than in builds from other branches.
#
# This assures that builds from other branches can only be deployed to App Store or Testflight,
# if version number is increased manually, i.e:
# REGULAR BUILDS TO BE DEPLOYED TO APP STORE MUST ALWAYS BE PRODUCED FROM PROUCTION BRANCH
#
if [ "`git rev-parse --abbrev-ref HEAD`" == "production_apple" ]
then
REVISION_BASE=0
else
REVISION_BASE=0
fi
let GIT_REVISION_CNT=${REVISION_BASE}+`git rev-list --count --first-parent HEAD`

echo "Using Git revision count for non-production build: ${GIT_REVISION_CNT}"

else

echo "Using GitLab Pipeline Id as revision count: ${CI_PIPELINE_ID}"
#
#   Pipeline id is increased for each build independent of source branch
#
let GIT_REVISION_CNT=${CI_PIPELINE_ID}

fi

GIT_REVISION_TAG=`git rev-parse --short --verify HEAD`

GIT_REVISION_NUM=${GIT_REVISION_CNT}${UNCOMMITTED_CHANGES}

#
# Extract version info from FastLibVersion.h
#
VERSION_INFO=`awk -f ${WORK_ROOT}/build/get_version.awk -- ${WORK_ROOT}/fcc/fvw/FastLibVersion.h`


#
# Update FVSourceRevision.h
#
# First write to temp file than move to FVSourceRevision.h because this script might be used concurrently in parallel builds
#
TMP_FILE="${WORK_ROOT}/fcc/fvw/$RANDOM.$$.tmp"

cat << EOF > ${TMP_FILE}
//
// This file is automatically created or updated with each build. Do not edit!
//
#define FV_SOURCE_REVISION          ${GIT_REVISION_NUM}
#define FV_SOURCE_REVISION_STR      "${GIT_REVISION_NUM}"
#define FV_VERSION_INFO             ${VERSION_INFO}
#define FV_VERSION_INFO_STR         "${VERSION_INFO}"
#define FV_FULL_VERSION_INFO        ${VERSION_INFO}.${GIT_REVISION_NUM}
#define FV_FULL_VERSION_INFO_STR    "${VERSION_INFO}.${GIT_REVISION_NUM}"
#define FV_GIT_COMMIT_STR           "${GIT_REVISION_TAG}"
EOF

mv ${TMP_FILE} ${WORK_ROOT}/fcc/fvw/FVSourceRevision.h


#
# Finally update plist file if regested 
#
if [ ! -z "${PLIST_FILE}" ] ; then

echo -n ${PLIST_FILE} \
| xargs -0 /usr/libexec/PlistBuddy -c "Set :CFBundleVersion ${VERSION_INFO}"

echo -n ${PLIST_FILE} \
| xargs -0 /usr/libexec/PlistBuddy -c "Set :CFBundleShortVersionString ${VERSION_INFO}"

echo -n  ${PLIST_FILE} \
| xargs -0 /usr/libexec/PlistBuddy -c "Set :com.fastviewer.source.revision ${GIT_REVISION_TAG}"

fi
