/*
 * Decompiled with CFR 0.152.
 */
package de.bdr.signme.api.v2;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;
import java.util.Objects;

public final class ECDSAKeyStore {
    private static final String KEYSTORE_TYPE = "pkcs12";
    private char[] keystorePw;
    private KeyStore keyStore;
    private X509Certificate certificate;
    private ECPublicKey publicKey;
    private ECPrivateKey privateKey;

    public ECDSAKeyStore(byte[] keyStoreBlob, char[] keystorePw, String keyPairAlias) throws IllegalArgumentException, IllegalStateException {
        Objects.requireNonNull(keyStoreBlob, "keyStoreBlob");
        Objects.requireNonNull(keystorePw, "keystorePw");
        Objects.requireNonNull(keyPairAlias, "keyPairAlias");
        try {
            this.keystorePw = (char[])keystorePw.clone();
            this.keyStore = this.loadKeyStore(keyStoreBlob, keystorePw);
            this.certificate = this.loadCertificate(keyPairAlias);
            this.publicKey = (ECPublicKey)this.certificate.getPublicKey();
            KeyStore.PrivateKeyEntry privKeyEntry = this.loadPrivateKeyEntry(keyPairAlias);
            this.privateKey = (ECPrivateKey)privKeyEntry.getPrivateKey();
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new IllegalStateException("creation of keystore does not work", e);
        }
    }

    public ECDSAKeyStore(File keyStoreFile, char[] keystorePw, String keyPairAlias) throws IllegalArgumentException, IllegalStateException {
        this(ECDSAKeyStore.loadKeyStoreFile(keyStoreFile), keystorePw, keyPairAlias);
    }

    private static byte[] loadKeyStoreFile(File keyStoreFile) throws IllegalArgumentException {
        Objects.requireNonNull(keyStoreFile, "keyStoreFile");
        if (!keyStoreFile.canRead()) {
            throw new IllegalArgumentException("keyStoreFile does not exist or cannot be read: " + keyStoreFile.getAbsolutePath());
        }
        try {
            return Files.readAllBytes(keyStoreFile.toPath());
        }
        catch (IOException e) {
            throw new IllegalArgumentException("problem loading keystore file from: " + keyStoreFile.getAbsolutePath());
        }
    }

    private KeyStore loadKeyStore(byte[] keyStoreBlob, char[] keystorePw) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
        try (ByteArrayInputStream is = new ByteArrayInputStream(keyStoreBlob);){
            keyStore.load(is, keystorePw);
        }
        return keyStore;
    }

    private X509Certificate loadCertificate(String keyPairAlias) {
        KeyStore.PrivateKeyEntry privKeyEntry = this.loadPrivateKeyEntry(keyPairAlias);
        return (X509Certificate)privKeyEntry.getCertificate();
    }

    private KeyStore.PrivateKeyEntry loadPrivateKeyEntry(String keyPairAlias) {
        KeyStore.Entry entry;
        Objects.requireNonNull(this.keyStore, "keyStore is already destroyed");
        Objects.requireNonNull(this.keystorePw, "keyStore is already destroyed");
        KeyStore.PasswordProtection pwProtForEntry = new KeyStore.PasswordProtection(this.keystorePw);
        try {
            entry = this.keyStore.getEntry(keyPairAlias, pwProtForEntry);
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException e) {
            throw new IllegalStateException("get key pair entry does not work", e);
        }
        if (entry == null || !(entry instanceof KeyStore.PrivateKeyEntry)) {
            throw new IllegalStateException("Keystore does not hold an key pair with alias=" + keyPairAlias);
        }
        KeyStore.PrivateKeyEntry privKeyEntry = (KeyStore.PrivateKeyEntry)entry;
        return privKeyEntry;
    }

    public X509Certificate getX509Certificate() {
        return this.certificate;
    }

    public ECPublicKey getPublicKey() {
        return this.publicKey;
    }

    ECPrivateKey getPrivate() {
        return this.privateKey;
    }

    public void destroy() {
        if (this.keystorePw != null) {
            Arrays.fill(this.keystorePw, '\u0000');
            this.keystorePw = null;
        }
        this.keyStore = null;
        this.certificate = null;
        this.publicKey = null;
        this.privateKey = null;
    }
}

