package de.intarsys.cloudsuite.gears.demo.auth;

import java.net.URI;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.security.auth.Subject;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.GSSManager;
import org.ietf.jgss.GSSName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.jaxrs.JaxrsTools;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;

public class KerberosContext extends CommonAuthContext {

	private static final Logger Log = LoggerFactory.getLogger(KerberosContext.class);

	public static final String AUTH_SCHEME_NEGOTIATE = "Negotiate";

	private GSSCredential delegCred;
	private GSSName srcName;

	public KerberosContext(KerberosModule authModule, Subject subject, IArgs args) {
		super(authModule, subject, args);
	}

	protected byte[] acceptToken(byte[] token) throws AuthenticationFailed {
		Configuration config = new Configuration() {

			@Override
			public AppConfigurationEntry[] getAppConfigurationEntry(String name) {
				Map<String, String> options = new HashMap<String, String>();
				options.put("principal", getAuthModule().getServicePrincipal());
				options.put("useKeyTab", "true");
				options.put("keyTab", getAuthModule().getKeyTabLocation());
				options.put("storeKey", "true");
				options.put("doNotPrompt", "true");
				options.put("isInitiator", "true");
				AppConfigurationEntry entry = new AppConfigurationEntry("com.sun.security.auth.module.Krb5LoginModule",
						AppConfigurationEntry.LoginModuleControlFlag.REQUIRED, options);
				return new AppConfigurationEntry[] { entry };
			}
		};
		try {
			LoginContext loginContext = new LoginContext(KerberosContext.class.getName(), null, null, config);
			loginContext.login();
			GSSManager manager = GSSManager.getInstance();
			GSSContext context = manager.createContext((GSSCredential) null);
			/*
			 * context state is initialized lazily on accept. Kerberos will need credentials
			 * to be able to decode the client token and it will try to find them in the
			 * current access control context's subject.
			 */
			PrivilegedExceptionAction<byte[]> action = () -> {
				return context.acceptSecContext(token, 0, token.length);
			};
			byte[] serverToken = Subject.doAs(loginContext.getSubject(), action);
			srcName = context.getSrcName();
			delegCred = context.getDelegCred();
			if (delegCred == null) {
				Log.warn("delegCred is null. Constrained delegation will not be available.");
			}
			return serverToken;
		} catch (GSSException | LoginException | PrivilegedActionException ex) {
			throw new AuthenticationFailed(ex);
		}
	}

	@Override
	public String authCallback(HttpServletRequest servletRequest) throws AuthenticationException {
		DemoPrincipal principal = new KerberosPrincipal(srcName.toString(), delegCred);
		getSubject().getPrincipals().add(principal);
		return principal.getToken();
	}

	@Override
	protected Response basicAuthenticate(HttpServletRequest servletRequest) throws AuthenticationException {
		byte[] serverToken = null;
		String authHeader = servletRequest.getHeader(HttpHeaders.AUTHORIZATION);
		if (authHeader != null) {
			Matcher matcher = Pattern.compile("(?i)" + AUTH_SCHEME_NEGOTIATE + "\\s+([a-zA-Z/+=0-9]+)\\s*")
					.matcher(authHeader);
			if (matcher.find()) {
				serverToken = acceptToken(Base64.getDecoder().decode(matcher.group(1)));
			}
		}

		if (serverToken == null) {
			// have browser do the negotiate
			return Response.status(Status.UNAUTHORIZED) //
					.header(HttpHeaders.WWW_AUTHENTICATE, AUTH_SCHEME_NEGOTIATE) //
					.build();
		}
		// proceed to callback
		URI uriCallback = JaxrsTools.getUriBuilderContext(servletRequest).path("/api/v1/authentication/callback") //
				.build();
		return Response.temporaryRedirect(uriCallback)
				.header(HttpHeaders.WWW_AUTHENTICATE,
						AUTH_SCHEME_NEGOTIATE + " " + Base64.getEncoder().encodeToString(serverToken)) //
				.build();
	}

	@Override
	protected void basicDeauthenticate() {
		getSubject().getPrincipals().removeIf((principal) -> principal instanceof KerberosPrincipal);
	}

	@Override
	public KerberosModule getAuthModule() {
		return (KerberosModule) super.getAuthModule();
	}
}
