package de.intarsys.cloudsuite.gears.demo.config;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.springframework.boot.web.servlet.ServletComponentScan;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.io.Resource;
import org.springframework.web.servlet.config.annotation.ViewControllerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.springframework.web.servlet.handler.SimpleUrlHandlerMapping;
import org.springframework.web.servlet.resource.PathResourceResolver;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;
import org.springframework.web.servlet.resource.ResourceTransformerChain;
import org.springframework.web.servlet.resource.TransformedResource;

import de.intarsys.tools.servlet.common.ServletTools;
import de.intarsys.tools.stream.StreamTools;
import jakarta.servlet.http.HttpServletRequest;

@Configuration
/*
 * we need standard jee servlet components
 */
@ServletComponentScan("de.intarsys")
public class WebApplicationConfig implements WebMvcConfigurer {

	@Override
	public void addViewControllers(ViewControllerRegistry registry) {
		registry.addViewController("/").setViewName("forward:/ng/index.html");
	}

	@Bean
	public SimpleUrlHandlerMapping handlerMapping() {
		SimpleUrlHandlerMapping mapping = new SimpleUrlHandlerMapping();
		mapping.setOrder(Ordered.HIGHEST_PRECEDENCE + 2);
		Map<String, Object> urlMap = new HashMap<>();
		urlMap.put("/ng/**", requestHandlerNg());
		mapping.setUrlMap(urlMap);
		return mapping;
	}

	@Bean
	public ResourceHttpRequestHandler requestHandlerNg() {
		ResourceHttpRequestHandler requestHandler = new ResourceHttpRequestHandler();
		requestHandler.setLocationValues(Arrays.asList("classpath:/static/ng/"));
		requestHandler.setResourceResolvers(Arrays.asList(new PathResourceResolver() {
			@Override
			protected Resource getResource(String resourcePath, Resource location) throws IOException {
				if (!resourcePath.contains(".")) {
					resourcePath = "index.html";
				}
				return super.getResource(resourcePath, location);
			}
		}));
		requestHandler.setResourceTransformers(Arrays.asList(this::transformNgIndex));
		return requestHandler;
	}

	protected Resource transformNgIndex(HttpServletRequest request, Resource resource,
			ResourceTransformerChain transformerChain)
			throws IOException {
		resource = transformerChain.transform(request, resource);

		String filename = resource.getFilename();
		if (!"index.html".equals(filename)) {
			return resource;
		}

		String content = StreamTools.getString(resource.getInputStream(), StandardCharsets.UTF_8);
		String baseHref = ServletTools.getRequestContextPath(request) + "/ng/";
		content = content.replaceAll("<base href=\"/.*/\">", "<base href=\"" + baseHref
				+ "\">");

		return new TransformedResource(resource, content.getBytes(StandardCharsets.UTF_8));
	}

}