/*
 *   o_
 * in|tarsys GmbH (c)
 *   
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.jaxrs;

import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.ResponseBuilder;
import jakarta.ws.rs.core.Response.Status;
import jakarta.ws.rs.ext.ExceptionMapper;
import jakarta.ws.rs.ext.Provider;

import de.intarsys.cloudsuite.gears.demo.auth.AuthenticationFailed;
import de.intarsys.tools.exception.ExceptionTools;
import de.intarsys.tools.jaxrs.exception.DtoError;
import de.intarsys.tools.jaxrs.exception.ResponseError;
import de.intarsys.tools.reflect.ClassTools;
import de.intarsys.tools.string.StringTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Provider
public final class DemoExceptionMapper implements ExceptionMapper<Exception> {

	private static final Logger Log = LoggerFactory.getLogger(DemoExceptionMapper.class);

	public static final String ERROR_SERVER = "ServerError";

	public static final String ERROR_CLIENT = "ClientError";

	public DemoExceptionMapper() {
	}

	protected Response createBadRequest(String code, Exception exception) {
		return createResponse(code, exception, Status.BAD_REQUEST);
	}

	protected Response createFromWebApplicationException(String code, WebApplicationException exception) {
		ResponseBuilder builder = Response.fromResponse(exception.getResponse());
		return createResponse(code, exception, builder);
	}

	protected Response createResponse(String code, Exception exception, Response.StatusType status) {
		return createResponse(code, exception, Response.status(status));
	}

	protected Response createResponse(String code, Exception exception, ResponseBuilder builder) {
		String tmpCode = code;
		if (StringTools.isEmpty(tmpCode)) {
			tmpCode = ClassTools.getUnqualifiedName(exception.getClass());
			if (tmpCode.endsWith("Exception")) {
				tmpCode = tmpCode.substring(0, tmpCode.length() - 9);
			}
		}
		ResponseError response = new ResponseError();
		DtoError detail = new DtoError();
		detail.setCode(tmpCode);
		detail.setMessage(ExceptionTools.getMessageShort(exception));
		response.setError(detail);
		return builder.entity(response).type(MediaType.APPLICATION_JSON).build();
	}

	protected Response createUnauthorized(String code, Exception exception) {
		return createResponse(code, exception, Status.UNAUTHORIZED);
	}

	@Override
	public Response toResponse(final Exception exception) {
		Log.warn(exception.getMessage(), exception);
		if (exception instanceof AuthenticationFailed) {
			return createUnauthorized(null, exception);
		} else if (exception instanceof IllegalArgumentException) {
			return createBadRequest(null, exception);
		} else if (exception instanceof WebApplicationException) {
			return createFromWebApplicationException(ERROR_SERVER, (WebApplicationException) exception);
		} else {
			return createBadRequest(ERROR_SERVER, exception);
		}
	}
}