/*
 *   o_
 * in|tarsys GmbH (c)
 *   
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.model;

import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriBuilder;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.annotation.SessionScope;

import de.intarsys.cloudsuite.gears.demo.auth.CommonAuthContext;
import de.intarsys.cloudsuite.gears.demo.auth.IAuthContext;
import de.intarsys.cloudsuite.gears.demo.auth.IAuthModule;
import de.intarsys.tools.exception.ExceptionTools;
import de.intarsys.tools.functor.Args;
import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.string.StringTools;

@Component
@SessionScope
public class AuthenticationBackend implements Serializable {

	private static final String PARAM_AUTH_ERROR = "auth_error";

	private static final String PARAM_AUTH_TOKEN = "auth_token";

	private static final String PARAM_AUTH_STATE = "auth_state";

	private static final String ARG_CALLBACK = "resume";

	private static final String ARG_STATE = "state";

	@Autowired(required = false)
	private List<IAuthModule> authModules = new ArrayList<>();

	private IAuthContext authContext;

	public AuthenticationBackend() {
	}

	/*
	 * a "landing page" for the authentication module callback.
	 * 
	 * This method will itself call back to the client provided callback,
	 * indicating success or failure.
	 */
	public Response callback(HttpServletRequest servletRequest) throws Exception {
		/*
		 * no hope
		 */
		if (getAuthContext() == null) {
			throw new IllegalStateException("no authentication context");
		}

		/*
		 * collect original arguments
		 */
		URI callbackUri = URI.create((String) getAuthContext().getArgs().get(ARG_CALLBACK));
		String state = (String) getAuthContext().getArgs().get(ARG_STATE);
		UriBuilder builder = UriBuilder //
				.fromUri(callbackUri) //
				.queryParam(PARAM_AUTH_STATE, state);

		/*
		 * get the token or failure
		 */
		try {
			String token = ((CommonAuthContext) getAuthContext()).authCallback(servletRequest);
			builder = builder.queryParam(PARAM_AUTH_TOKEN, token);
		} catch (Exception e) {
			builder = builder.queryParam(PARAM_AUTH_ERROR, ExceptionTools.getMessage(e));
		}
		return Response.temporaryRedirect(builder.build()).build();
	}

	/*
	 * forward authentication to another module
	 */
	public Response forward(HttpServletRequest servletRequest, String id) throws Exception {
		IAuthContext context = getAuthContext();
		if (context == null) {
			throw new IllegalStateException("no authentication context");
		}
		IAuthModule module = getAuthModule(id);
		IArgs args = context.getArgs();
		setAuthContext(module.createAuthContext(args));
		return getAuthContext().authenticate(servletRequest);
	}

	public IAuthContext getAuthContext() {
		return authContext;
	}

	protected IAuthModule getAuthModule(String id) {
		return getAuthModules().stream().filter((a) -> id.equals(a.getId())).findAny()
				.orElseThrow(() -> new IllegalArgumentException("unknown auth module " + id));
	}

	public List<IAuthModule> getAuthModules() {
		return authModules;
	}

	/*
	 * clear authentication state
	 */
	public void logout() throws Exception {
		if (getAuthContext() != null) {
			getAuthContext().deauthenticate();
		}
		setAuthContext(null);
	}

	/*
	 * start authentication using a dedicated module
	 */
	public Response request(HttpServletRequest servletRequest, String id, String callback, String state)
			throws Exception {
		if (StringTools.isEmpty(id)) {
			throw new IllegalArgumentException("id required");
		}
		if (StringTools.isEmpty(callback)) {
			throw new IllegalArgumentException("callback required");
		}
		if (StringTools.isEmpty(state)) {
			throw new IllegalArgumentException("state required");
		}
		IAuthModule module = getAuthModule(id);
		IArgs args = Args.create();
		args.put(ARG_CALLBACK, callback);
		args.put(ARG_STATE, state);
		setAuthContext(module.createAuthContext(args));
		return getAuthContext().authenticate(servletRequest);
	}

	public void setAuthContext(IAuthContext authContext) {
		this.authContext = authContext;
	}

	public void setAuthModules(List<IAuthModule> authModules) {
		this.authModules = authModules;
	}

}
