/*
 *   o_
 * in|tarsys GmbH (c)
 *
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.service;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

import org.glassfish.jersey.media.multipart.ContentDisposition;
import org.glassfish.jersey.media.multipart.FormDataBodyPart;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.intarsys.cloudsuite.gears.demo.jaxrs.Authenticated;
import de.intarsys.cloudsuite.gears.demo.model.DemoBackend;
import de.intarsys.cloudsuite.gears.demo.model.DemoPreset;
import de.intarsys.conversation.service.client.api.RequestAcknowledge;
import de.intarsys.conversation.service.client.api.RequestCancel;
import de.intarsys.conversation.service.client.api.ResponseCancel;
import de.intarsys.tools.locator.StreamLocator;
import de.intarsys.tools.stream.StreamTools;
import jakarta.annotation.PostConstruct;
import jakarta.inject.Inject;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.Context;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.SecurityContext;

@Authenticated
@Path("/v1/demo")
public class SvcDemo {

	private static final Logger Log = LoggerFactory.getLogger("demo");

	@Inject
	private transient DemoBackend demoBackend;

	@Inject
	private SecurityContext securityContext;

	@Context
	private HttpServletRequest httpServletRequest;

	@POST
	@Path("/conversation/acknowledge")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation conversationAcknowledge(RequestAcknowledge request) throws Exception {
		return getDemoBackend().conversationAcknowledge(request, httpServletRequest);
	}

	@POST
	@Path("/conversation/cancel")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseCancel conversationCancel(RequestCancel request) throws Exception {
		return getDemoBackend().conversationCancel(request, httpServletRequest);
	}

	@POST
	@Path("/deleteAllDocuments")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response deleteAllDocuments() throws IOException {
		getDemoBackend().deleteAllDocuments();
		return Response.noContent().build();
	}

	@POST
	@Path("/deleteDocument")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response deleteDocument(RequestDemoDeleteDocument request) throws IOException {
		getDemoBackend().deleteDocument(request.getDocument().getName());
		return Response.noContent().build();
	}

	@POST
	@Path("/deleteDocuments")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response deleteDocuments(RequestDemoDeleteDocuments request) throws IOException {
		getDemoBackend().deleteDocuments(request.getDocuments());
		return Response.noContent().build();
	}

	@POST
	@Path("/downloadDocument")
	@Consumes(MediaType.APPLICATION_JSON)
	public Response downloadDocument(RequestDemoDownloadDocument request) throws Exception {
		return getDemoBackend().downloadDocument(request.getDocument());
	}

	@POST
	@Path("/downloadDocuments")
	@Consumes(MediaType.APPLICATION_JSON)
	public Response downloadDocuments(RequestDemoDownloadDocuments request) throws Exception {
		return getDemoBackend().downloadDocuments(request.getDocuments());
	}

	@POST
	@Path("/editDocument")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response editDocument(RequestDemoOpenDocument request) throws Exception {
		getDemoBackend().editDocument(request.getDocument());
		return Response //
				.noContent() //
				.build();
	}

	protected DemoBackend getDemoBackend() {
		return demoBackend;
	}

	@POST
	@Path("/getDocuments")
	@Produces(MediaType.APPLICATION_JSON)
	public List<DtoDocState> getDocuments() {
		return getDemoBackend().getDocuments().stream().map(DtoDocState::new).toList();
	}

	@POST
	@Path("/getPresets")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public List<DemoPreset> getPresets() throws Exception {
		return getDemoBackend().getPresets();
	}

	@POST
	@Path("/getSelectedPreset")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseGetSelectedPreset getSelectedPreset() throws Exception {
		return new ResponseGetSelectedPreset(getDemoBackend().getSelectedPreset());
	}

	@PostConstruct
	public void init() {
		getDemoBackend().setSecurityContext(securityContext);
	}

	@POST
	@Path("/openDocument")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation openDocument(RequestDemoOpenDocument request) throws Exception {
		return getDemoBackend().openDocument(request.getDocument(), request.getRedirectUri());
	}

	@POST
	@Path("/openDocuments")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation openDocuments(RequestDemoOpenDocuments request) throws Exception {
		return getDemoBackend().openDocuments(request.getDocuments(), request.getRedirectUri());
	}

	@POST
	@Path("/setPresets")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response setPresets(List<DemoPreset> request) throws Exception {
		getDemoBackend().setPresets(request);
		return Response //
				.noContent() //
				.build();
	}

	@POST
	@Path("/setSelectedPreset")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response setSelectedPreset(RequestSelectPreset request) throws Exception {
		getDemoBackend().setSelectedPreset(request.getId());
		return Response //
				.noContent() //
				.build();
	}

	@POST
	@Path("/signDocument")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation signDocument(RequestDemoSignDocument request) throws Exception {
		return getDemoBackend().signDocument(request.getDocument(), request.getRedirectUri());
	}

	@POST
	@Path("/signDocuments")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation signDocuments(RequestDemoSignDocuments request) throws Exception {
		return getDemoBackend().signDocuments(request.getDocuments(), request.getRedirectUri());
	}

	@POST
	@Path("/timestampDocument")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation timestampDocument(RequestDemoSignDocument request) throws Exception {
		return getDemoBackend().timestampDocument(request.getDocument(), request.getRedirectUri());
	}

	@POST
	@Path("/timestampDocuments")
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public ResponseDemoConversation timestampDocuments(RequestDemoSignDocuments request) throws Exception {
		return getDemoBackend().timestampDocuments(request.getDocuments(), request.getRedirectUri());
	}

	@POST
	@Path("/uploadDocument")
	@Consumes(MediaType.MULTIPART_FORM_DATA)
	@Produces(MediaType.APPLICATION_JSON)
	public Response uploadDocument(FormDataMultiPart body) throws IOException {
		Map<String, List<FormDataBodyPart>> fields = body.getFields();
		for (List<FormDataBodyPart> parts : fields.values()) {
			for (FormDataBodyPart part : parts) {
				ContentDisposition disposition = part.getContentDisposition();
				InputStream is = part.getEntityAs(InputStream.class);
				try {
					String filename = disposition.getFileName();
					/*-
					 * this is a workaround for underspecified form data file transmission
					 * - JavaScript FormData can not specify the new "filename*" parameter
					 * - Browsers transmit normally "filename" parameter *not* in 7bit
					 * - Jersey multipart implementation is not able to parse this
					 * - so we "transcode" by hand...
					 */
					filename = new String(filename.getBytes(StandardCharsets.ISO_8859_1), StandardCharsets.UTF_8);
					Log.info("found attachment {}, {}", part.getName(), filename);
					getDemoBackend().importDocument(filename, new StreamLocator(is, "upload.data"));
				} finally {
					StreamTools.close(is);
				}
			}
		}
		return Response //
				.noContent() //
				.build();
	}

}
