import { Component, ElementRef, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { MatDialog, MatDialogConfig } from '@angular/material/dialog';
import { DomSanitizer, SafeResourceUrl } from '@angular/platform-browser';
import { ActivatedRoute, Router } from '@angular/router';
import { JsonEditorOptions } from 'ang-jsoneditor';

import { PromptComponent } from '../prompt/prompt.component';

@Component({
  selector: 'app-embedded-view',
  templateUrl: './embedded-view.component.html',
  styleUrls: ['./embedded-view.component.css']
})
export class EmbeddedViewComponent implements OnInit, OnDestroy {

  constructor(
    private route: ActivatedRoute,
    private router: Router,
    private sanitizer: DomSanitizer,
    private dialog: MatDialog
  ) {
    this.editorOptions = new JsonEditorOptions();
    this.editorOptions.modes = ['view'];
    this.editorOptions.mode = 'view';
    this.editorOptions.search = false;
  }

  urlString: string;

  conversation: string;

  url: SafeResourceUrl;

  editorOptions: JsonEditorOptions;
  counter = 0;

  @ViewChild('iframe', { read: ElementRef }) tref: ElementRef;

  messages: any[] = [];

  onMessage = ((message) => { this.receiveMessage(message); }).bind(this);

  ngOnInit() {
    this.conversation = this.route.snapshot.queryParamMap.get('conversation');
    this.urlString = this.route.snapshot.queryParamMap.get('url');
    this.url = this.sanitizer.bypassSecurityTrustResourceUrl(this.urlString);
    window.addEventListener('message', this.onMessage);
  }

  ngOnDestroy() {
    window.removeEventListener('message', this.onMessage);
  }

  postMessage(message: any) {
    this.tref.nativeElement.contentWindow.postMessage(message, this.urlString);
  }

  receiveMessage(message: any) {
    const data = message.data;
    this.messages = [data, ... this.messages];
    if (data.type === 'reply') {
      if (data.id === 'ok' || data.id === 'cancel') {
        this.router.navigate(['/app', 'documents'], { queryParams: { cs_conversation: this.conversation } });
      }
    }
    if (data.type === 'event') {
    }
  }

  onEcho() {
    this.postMessage({
      type: 'trigger',
      id: 'echo',
      sendResult: true,
      action: {
        type: "Echo",
        counter: this.counter,
        message: "call " + this.counter,
      }
    });
    this.counter++;
  }

  onThrow() {
    this.postMessage({
      type: 'trigger',
      id: 'throw',
      action: {
        type: "Throw",
        code: "forced",
        message: "test throw"
      }
    });
    this.counter++;
  }

  onSign() {
    this.postMessage({
      type: 'trigger',
      id: 'sign',
      widget: 'my.widget.sign'
    });
  }

  onOk() {
    this.postMessage({
      type: 'trigger',
      id: 'ok',
      action: {
        type: "Ok"
      }
    });
  }

  onCancel() {
    this.postMessage({
      type: 'trigger',
      id: 'cancel',
      action: {
        type: "Cancel"
      }
    });
  }

  onCall() {
    const dialogConfig = new MatDialogConfig();
    dialogConfig.autoFocus = true;
    dialogConfig.minWidth = "50vw";
    const dialogRef = this.dialog.open(PromptComponent, dialogConfig);
    dialogRef.afterClosed().subscribe(input => {
      if (input) {
        const action = JSON.parse(input);
        this.postMessage({
          type: 'trigger',
          id: 'call',
          sendResult: true,
          action: action,
        });
      }
    });
  }

  onClear() {
    this.messages = [];
  }

}
