import { Location } from '@angular/common';
import { HttpClient, HttpErrorResponse, HttpHeaders } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { saveAs } from 'file-saver';
import { BlockUI, NgBlockUI } from 'ng-block-ui';
import { Observable, throwError } from 'rxjs';
import { catchError, finalize, map, mergeMap } from 'rxjs/operators';

import { AppError } from '../error/apperror.model';

import * as metainfo from '../meta.json';

declare var ConversationHandler: any;

@Injectable()
export class RemoteService {

  @BlockUI() blockUI: NgBlockUI;

  public urlServer;

  private urlPrefix = '/api/v1';

  constructor(
    private location: Location,
    private http: HttpClient
  ) {
    const searchParams = new URL(window.location.href).searchParams;
    this.urlServer = searchParams.get('urlServer');
    if (!this.urlServer || this.urlServer === '') {
      this.urlServer = localStorage.getItem(metainfo.default['id'] + '.urlServer');
    }
    if (!this.urlServer || this.urlServer === '') {
      this.urlServer = '..';
    }
    localStorage.setItem(metainfo.default['id'] + '.urlServer', this.urlServer);
  }

  toExternalUrl(path: string) {
    return new URL(window.location.origin + this.location.prepareExternalUrl(path)).toString();
  }

  toTargetUrl(path: string) {
    return Location.joinWithSlash(Location.joinWithSlash(this.urlServer, this.urlPrefix), path);
  }

  contextUrl() {
    return this.urlServer;
  }

  fetch(url: any, request: any, blockOptions: any = null): Observable<any> {
    if (blockOptions && blockOptions.message) {
      this.blockUI.reset();
      this.blockUI.start(blockOptions.message);
    }
    return this.http
      .request('POST', url, {
        body: request,
        withCredentials: true,
        headers: new HttpHeaders({
          'Content-Type': 'application/json'
        })
      })
      .pipe(
        catchError((error) => throwError(this.mapError(error))),
        finalize(() => {
          if (blockOptions && blockOptions.message) {
            this.blockUI.stop();
          }
        })
      );
  }

  fetchSnapshot(url: any, request: any, blockOptions: any = null): Observable<any> {
    return this
      .fetch(url, request, blockOptions)
      .pipe(
        map((result) => result.snapshot)
      );
  }

  fetchConversational(url: any, request: any, blockOptions: any, context: any = {}): Observable<any> {
    return this.fetchSnapshot(url, request, blockOptions)
      .pipe(
        mergeMap((snapshot) => ConversationHandler.active.handle(snapshot, context))
      );
  }

  download(url: any, request: any, blockOptions: any = null): Observable<void> {
    if (blockOptions && blockOptions.message) {
      this.blockUI.reset();
      this.blockUI.start(blockOptions.message);
    }
    return this.http
      .request('POST', url, {
        body: request,
        withCredentials: true,
        observe: 'response',
        responseType: 'blob'
      })
      .pipe(
        map((result) => {
          const contentDisposition = result.headers.get('Content-Disposition');
          const filename = /filename="([^"\\]*(?:\\.[^"\\]*)*)"/i.exec(contentDisposition)[1];
          saveAs(result.body, filename, false);
          return null;
        }),
        catchError((error) => throwError(this.mapError(error))),
        finalize(() => {
          if (blockOptions && blockOptions.message) {
            this.blockUI.stop();
          }
        })
      );
  }

  private mapError(error: any): any {
    let reason = error.message || error;
    if (error instanceof HttpErrorResponse) {
      let code = null;
      let message = null;
      if (error.error && error.error._error) {
        code = error.error._error.code;
        message = error.error._error.message;
      }
      if (error.status === 400) {
        reason = new AppError(code ? code : 'BadRequest', message ? message : 'Bad or missing parameter');
      } else if (error.status === 401) {
        reason = new AppError('Unauthorized', 'Authentication required');
      } else if (error.status === 402) {
        reason = new AppError('PaymentRequired', 'No license or limit exceeded');
      } else if (error.status === 403) {
        reason = new AppError('Forbidden', 'Permission required');
      } else if (error.status === 404) {
        reason = new AppError('NotFound', 'Resource not found');
      } else if (error.status === 405) {
        reason = new AppError('MethodNotAllowed', 'Invalid call syntax (HTTP method)');
      } else if (error.status === 406) {
        reason = new AppError('NotAcceptable', 'Invalid call syntax (content)');
      } else if (error.status === 407) {
        reason = new AppError('ProxyAuthenticationRequired', 'Authentication required');
      } else if (error.status === 408) {
        reason = new AppError('RequestTimeout', 'Request timeout, try again later.');
      } else if (error.status === 409) {
        reason = new AppError('Conflict', 'Resource already changed');
      } else if (error.status === 410) {
        reason = new AppError('Gone', 'Resource no longer available');
      } else if (error.status === 411) {
        reason = new AppError('LengthRequired', 'Invalid call syntax (length)');
      } else if (error.status === 496) { // nginx proprietary standard
        reason = new AppError('Unauthorized', 'SSL Authentication required');
      } else if (error.status === 500) {
        reason = new AppError(code ? code : 'InternalServerError', message ? message : 'Unexpected server error');
      } else if (error.status === 501) {
        reason = new AppError('NotImplemented', 'Feature not available');
      } else if (error.status === 503) {
        reason = new AppError('ServiceUnavailable', 'Service temporarily unavailable');
      } else if (error.status === 0) {
        reason = new AppError('ConnectionFailed', 'Server not responding. Check URL.');
      } else {
        reason = new AppError('Error', 'Unexpected error ' + reason);
      }
    }
    return reason;
  }
}
