import { Injectable } from '@angular/core';
import { Observable, of, throwError } from 'rxjs';
import { catchError, mergeMap, tap } from 'rxjs/operators';

import { AuthService } from '../auth/auth.service';
import { NoticeService } from '../notice/notice.service';
import { RemoteService } from '../service/remote.service';

@Injectable()
export class SettingsService {

  constructor(
    private authService: AuthService,
    private remoteService: RemoteService,
    private noticeService: NoticeService
  ) {
  }

  presets: any[] = null;

  preset: any;

  presetId: string;

  load(): Observable<any> {
    if (this.authService.isAuthenticated() && this.presets) {
      return of(null);
    }
    let urlTarget = this.remoteService.toTargetUrl('demo/getPresets');
    let request = {};
    return this.remoteService
      .fetch(urlTarget, request, { message: 'Loading...' })
      .pipe(
        mergeMap(result => {
          this.presets = result;
          urlTarget = this.remoteService.toTargetUrl('demo/getSelectedPreset');
          request = {};
          return this.remoteService
            .fetch(urlTarget, request, { message: 'Loading...' });
        }),
        mergeMap(result => {
          this.presetId = result.id;
          this.checkPresets();
          return of(null);
        }),
        catchError(error => {
          this.noticeService.info('Settings not available', 'Service info');
          this.presets = null;
          this.presetId = null;
          return throwError(error);
        }),
        tap(result => {
          this.noticeService.info('Settings loaded', 'Service info');
        })
      );
  }

  checkPresets() {
    if (!this.presetId) {
      this.preset = this.presets.find(preset => true);
      if (this.preset) {
        this.presetId = this.preset.id;
      }
    } else {
      this.preset = this.presets.find(preset => this.presetId === preset.id);
    }
    if (!this.preset) {
      this.presetId = "default";
      this.preset = this.presets.find(preset => this.presetId === preset.id);
    }
    if (!this.preset) {
      this.createPreset("default");
    }
  }

  createPreset(id: string) {
    const newSettings = this.preset ? { ... this.preset.settings } : {};
    this.preset = {
      id: id,
      settings: newSettings
    };
    this.presets.push(this.preset);
    this.presetId = id;
    return this
      .callSetPresets()
      .pipe(
        mergeMap(result => this.callSetPresetId())
      );
  }

  removePreset(aPreset: any) {
    this.presets = this.presets.filter(preset => aPreset !== preset);
    this.presetId = null;
    this.checkPresets();
    return this
      .callSetPresets()
      .pipe(
        mergeMap(result => this.callSetPresetId())
      );
  }

  save(newPresets: any[]): Observable<any> {
    this.presets = newPresets;
    this.checkPresets();
    return this
      .callSetPresets()
      .pipe(
        mergeMap(result => this.callSetPresetId())
      );
  }

  setPresetId(id: string): Observable<any> {
    this.presetId = id;
    this.checkPresets();
    return this
      .callSetPresets()
      .pipe(
        mergeMap(result => this.callSetPresetId())
      );
  }

  callSetPresets(): Observable<any> {
    const urlTarget = this.remoteService.toTargetUrl('demo/setPresets');
    const request = this.presets;
    return this.remoteService
      .fetch(urlTarget, request, { message: 'Saving...' });
  }

  callSetPresetId(): Observable<any> {
    const urlTarget = this.remoteService.toTargetUrl('demo/setSelectedPreset');
    const request = { id: this.presetId };
    return this.remoteService
      .fetch(urlTarget, request, { message: 'Saving...' });
  }
}
