/*
 *   o_
 * in|tarsys GmbH (c)
 *
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.core.service.common.api;

import java.io.File;
import java.io.IOException;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportDocumentLocatorContentDeserializer;
import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportDocumentLocatorContentSerializer;
import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportDocumentLocatorPathDeserializer;
import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportItemLocatorHandleDeserializer;
import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportItemLocatorHandleSerializer;
import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportItemLocatorPathSerializer;
import de.intarsys.tools.jaxrs.ServiceParam;
import de.intarsys.tools.lang.LangTools;
import de.intarsys.tools.locator.ByteArrayLocator;
import de.intarsys.tools.locator.FileLocator;
import de.intarsys.tools.locator.ILocator;
import de.intarsys.tools.string.StringTools;

/**
 * A "document" type item to be transported to or from a cloudsuite service.
 */
@ServiceParam
public class TransportDocument extends TransportItem {

	public static class TransportDocumentBuilder extends TransportItemBuilder<TransportDocument> {
		protected String contentType;
		protected ILocator content;
		protected ILocator handle;
		protected ILocator path;
		protected Long length;

		@Override
		public TransportDocument build() {
			TransportDocument result = super.build();
			if (contentType != null) {
				result.setContentType(contentType);
			}

			if (content != null) {
				result.setContent(content);
			}

			if (path != null) {
				result.setPath(path);
			}

			if (handle != null) {
				result.setHandle(handle);
			}

			if (length != null) {
				result.setLength(length);
			}

			return result;
		}

		public TransportDocumentBuilder contentType(String contentType) {
			this.contentType = contentType;
			return this;
		}

		public TransportDocumentBuilder content(ILocator content) {
			this.content = content;
			return this;
		}

		@Override
		protected TransportDocument createResult() {
			return new TransportDocument();
		}

		public TransportDocumentBuilder handle(ILocator handle) {
			this.handle = handle;
			return this;
		}

		@Override
		public TransportDocumentBuilder name(String name) {
			return (TransportDocumentBuilder) super.name(name);
		}

		public TransportDocumentBuilder path(ILocator locator) {
			this.path = locator;
			return this;
		}

		public TransportDocumentBuilder length(Long length) {
			this.length = length;
			return this;
		}

		@Override
		public TransportDocumentBuilder property(String name, Object value) {
			return (TransportDocumentBuilder) super.property(name, value);
		}
	}

	public static TransportDocumentBuilder builder() {
		return new TransportDocumentBuilder();
	}

	@JsonInclude(Include.NON_NULL)
	private String contentType;

	@JsonSerialize(using = TransportDocumentLocatorContentSerializer.class)
	@JsonDeserialize(using = TransportDocumentLocatorContentDeserializer.class)
	@JsonInclude(Include.NON_NULL)
	private ILocator content;

	@JsonSerialize(using = TransportItemLocatorPathSerializer.class)
	@JsonDeserialize(using = TransportDocumentLocatorPathDeserializer.class)
	@JsonInclude(Include.NON_NULL)
	private ILocator path;

	@JsonSerialize(using = TransportItemLocatorHandleSerializer.class)
	@JsonDeserialize(using = TransportItemLocatorHandleDeserializer.class)
	@JsonInclude(Include.NON_NULL)
	private ILocator handle;

	@JsonInclude(Include.NON_NULL)
	private Long length;

	@JsonIgnore
	private ILocator locator;

	public TransportDocument() {
		super();
	}

	public TransportDocument(File content) {
		super();
		setContent(new FileLocator(content));
	}

	public TransportDocument(ILocator content) {
		super();
		setContent(content);
	}

	public TransportDocument(String name) {
		super(name);
	}

	@Override
	public boolean equals(Object other) {
		if (!(other instanceof TransportDocument)) {
			return false;
		}
		TransportDocument otherDoc = (TransportDocument) other;
		if (!LangTools.equals(locator, otherDoc.locator)) {
			return false;
		}
		if (!LangTools.equals(getProperties(), otherDoc.getProperties())) {
			return false;
		}
		return true;
	}

	protected ILocator getContent() {
		return content;
	}

	public String getContentType() {
		return contentType;
	}

	public ILocator getHandle() {
		return handle;
	}

	public Long getLength() {
		return length;
	}

	@Override
	public ILocator getLocator() {
		if (locator == null) {
			return new ByteArrayLocator(new byte[0], getName());
		}
		return locator;
	}

	protected ILocator getPath() {
		return path;
	}

	@Override
	public String getType() {
		return "d";
	}

	@Override
	public boolean isDocument() {
		return true;
	}

	@Override
	public boolean isLink() {
		return handle != null;
	}

	@Override
	public void setContent(ILocator locator) {
		this.content = locator;
		setLocator(locator);
	}

	public void setContentType(String contentType) {
		this.contentType = contentType;
	}

	@Override
	public void setHandle(ILocator locator) {
		this.handle = locator;
		setLocator(locator);
		try {
			setLength(locator.getLength());
		} catch (IOException e) {
			// ignore
		}
	}

	public void setLength(Long length) {
		this.length = length;
	}

	protected void setLocator(ILocator locator) {
		if (StringTools.isEmpty(basicGetName()) && locator != null) {
			setName(locator.getName());
		}
		this.locator = locator;
	}

	@Override
	public void setPath(ILocator locator) {
		this.path = locator;
		setLocator(locator);
	}
}
