/*
 *   o_
 * in|tarsys GmbH (c)
 *   
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.core.service.common.api;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportItemDeserializer;
import de.intarsys.tools.attribute.AttributeMap;
import de.intarsys.tools.functor.ArgTools;
import de.intarsys.tools.functor.Args;
import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.jackson.ArgsToJavaConverter;
import de.intarsys.tools.jackson.JavaToArgsConverter;
import de.intarsys.tools.jaxrs.ServiceParam;
import de.intarsys.tools.locator.ILocator;
import de.intarsys.tools.string.StringTools;

/**
 * An abstract item to be transported to or from a cloud suite service.
 *
 */
@ServiceParam
@JsonDeserialize(using = TransportItemDeserializer.class)
@JsonPropertyOrder({ "type", "name" })
public abstract class TransportItem {

	public abstract static class TransportItemBuilder<R extends TransportItem> {

		private IArgs properties = Args.create();

		private String name;

		public R build() {
			R result = createResult();
			result.setProperties(properties);
			result.setName(name);
			return result;
		}

		protected abstract R createResult();

		public TransportItemBuilder<R> name(String name) {
			this.name = name;
			return this;
		}

		public TransportItemBuilder<R> property(String name, Object value) {
			ArgTools.putPath(properties, name, value);
			return this;
		}

	}

	private String name;

	@JsonIgnore
	private final AttributeMap attributes = new AttributeMap();

	@JsonInclude(Include.NON_EMPTY)
	@JsonSerialize(converter = ArgsToJavaConverter.class)
	@JsonDeserialize(converter = JavaToArgsConverter.class)
	private IArgs properties = Args.create();

	@JsonIgnore
	private TransportFolder parent;

	public TransportItem() {
	}

	public TransportItem(String name) {
		super();
		this.name = name;
	}

	public void addProperties(IArgs properties) {
		ArgTools.putAllDeep(this.properties, properties);
	}

	protected String basicGetName() {
		return name;
	}

	@JsonIgnore
	public Object getAttribute(Object key) {
		return attributes.getAttribute(key);
	}

	@JsonIgnore
	public abstract ILocator getLocator();

	public String getName() {
		if (StringTools.isEmpty(name)) {
			return "undefined";
		}
		return name;
	}

	public TransportFolder getParent() {
		return parent;
	}

	public IArgs getProperties() {
		return properties;
	}

	public Object getProperty(String name) {
		return ArgTools.getObject(properties, name, null);
	}

	@JsonProperty("type")
	public String getType() {
		return "";
	}

	@JsonIgnore
	public boolean isDocument() {
		return false;
	}

	@JsonIgnore
	public boolean isFolder() {
		return false;
	}

	@JsonIgnore
	public boolean isLink() {
		return false;
	}

	@JsonIgnore
	public Object removeAttribute(Object key) {
		return attributes.removeAttribute(key);
	}

	@JsonIgnore
	public Object setAttribute(Object key, Object value) {
		return attributes.setAttribute(key, value);
	}

	public abstract void setContent(ILocator locator);

	public abstract void setHandle(ILocator locator);

	public void setName(String name) {
		this.name = name;
	}

	public void setParent(TransportFolder parent) {
		this.parent = parent;
	}

	public abstract void setPath(ILocator locator);

	public void setProperties(IArgs properties) {
		this.properties = properties;
	}

	public void setProperty(String name, Object value) {
		ArgTools.putPath(properties, name, value);
	}

}
