/*
 *   o_
 * in|tarsys GmbH (c)
 *   
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.core.service.common.jackson;

import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonStreamContext;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanDescription;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.deser.ResolvableDeserializer;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

import de.intarsys.cloudsuite.gears.core.service.common.api.TransportDocument;
import de.intarsys.cloudsuite.gears.core.service.common.api.TransportFolder;
import de.intarsys.cloudsuite.gears.core.service.common.api.TransportItem;

/**
 * The jackson deserialization implementation for {@link TransportItem}.
 * 
 * While in practice you can use the "plain" deserialization for this data
 * transfer type, the optimizations supported by this deserializer are crucial
 * for scaling the service.
 * 
 * The deserializer enforces that "type" and "name" properties *must* be
 * available first. This restriction allows for early TransportItem creation
 * without buffering the content.
 *
 * The callbacks in addition allow that the real backend objects (like the
 * repository we put the content into) can be involved in the marshalling and
 * streaming process. Multiple copies and crypto operations should be
 * unnecessary.
 */
public class TransportItemDeserializer extends StdDeserializer<TransportItem> {

	private static final String FIELD_NAME = "name";

	private static final String FIELD_TYPE = "type";

	private static final Logger Log = LoggerFactory.getLogger(TransportItemDeserializer.class);

	private final ITransportItemDeserializerCallback callback;

	public TransportItemDeserializer() {
		super(TransportItem.class);
		this.callback = null;
	}

	public TransportItemDeserializer(ITransportItemDeserializerCallback callback) {
		super(TransportItem.class);
		this.callback = callback;
	}

	protected void assignParent(JsonParser parser, TransportItem item) {
		JsonStreamContext context = parser.getParsingContext();
		if (context != null) {
			// if contained, this is List parsing context
			context = context.getParent();
		}
		if (context != null) {
			// if contained, this is TransportFolder parsing context
			context = context.getParent();
		}
		if (context != null) {
			Object parentCandidate = context.getCurrentValue();
			if (parentCandidate instanceof TransportFolder) {
				item.setParent((TransportFolder) parentCandidate);
			}
		}
	}

	protected JsonDeserializer createDeserializer(DeserializationContext ctxt, Class clazz) throws IOException {
		JavaType type = ctxt.getTypeFactory().constructType(clazz);
		BeanDescription beanDesc = ctxt.getConfig().introspect(type);
		JsonDeserializer deser = ctxt.getFactory().createBeanDeserializer(ctxt, type, beanDesc);
		if (deser instanceof ResolvableDeserializer) {
			((ResolvableDeserializer) deser).resolve(ctxt);
		}
		return deser;
	}

	@Override
	public TransportItem deserialize(JsonParser parser, DeserializationContext ctxt) throws IOException {
		JsonToken token;
		token = parser.getCurrentToken();
		if (token == JsonToken.VALUE_NULL) {
			return null;
		}
		if (token == JsonToken.START_OBJECT) {
			if (Log.isEnabledForLevel(Level.TRACE)) {
				Log.trace("memory: total {}, free {}", Runtime.getRuntime().totalMemory(),
						Runtime.getRuntime().freeMemory());
			}
			String tokenText;
			token = parser.nextToken();
			tokenText = parser.getText();
			if (FIELD_TYPE.equals(tokenText)) {
				parser.nextToken();
				String typeId = parser.getText();
				if ("d".equals(typeId)) {
					parser.nextToken();
					return deserializeDocument(parser, ctxt);
				} else if ("f".equals(typeId)) {
					parser.nextToken();
					return deserializeFolder(parser, ctxt);
				} else {
					throw new JsonMappingException(parser, "'type' " + typeId + " not supported");
				}
			} else if (FIELD_NAME.equals(tokenText)) {
				return deserializeDocument(parser, ctxt);
			} else {
				throw new JsonMappingException(parser, "'type' required as first element, found " + tokenText);
			}
		}
		throw new JsonMappingException(parser, "unsupported token " + token);
	}

	protected TransportItem deserializeDocument(JsonParser parser, DeserializationContext ctxt) throws IOException {
		String tokenText;

		JsonDeserializer deser = createDeserializer(ctxt, TransportDocument.class);
		// JsonDeserializer deser = ctxt.findContextualValueDeserializer(type,
		// null);
		tokenText = parser.getText();
		if (FIELD_NAME.equals(tokenText)) {
			parser.nextToken();
			tokenText = parser.getText();
			parser.nextToken();
			TransportDocument result = new TransportDocument(tokenText);
			parser.setCurrentValue(result);
			assignParent(parser, result);
			if (callback != null) {
				callback.transportDocumentCreated(result, parser, ctxt);
			}
			Log.debug("creating document {}", result);
			deser.deserialize(parser, ctxt, result);
			if (callback != null) {
				callback.transportDocumentDeserialized(result, parser, ctxt);
			}
			return result;
		}
		throw new JsonMappingException(parser, "'name' required as second element");
	}

	protected TransportFolder deserializeFolder(JsonParser parser, DeserializationContext ctxt) throws IOException {
		String tokenText;

		JsonDeserializer deser = createDeserializer(ctxt, TransportFolder.class);
		tokenText = parser.getText();
		if (FIELD_NAME.equals(tokenText)) {
			parser.nextToken();
			tokenText = parser.getText();
			parser.nextToken();
			TransportFolder result = new TransportFolder(tokenText);
			parser.setCurrentValue(result);
			assignParent(parser, result);
			Log.debug("creating folder {}", result);
			if (callback != null) {
				callback.transportFolderCreated(result, parser, ctxt);
			}
			deser.deserialize(parser, ctxt, result);
			if (callback != null) {
				callback.transportFolderDeserialized(result, parser, ctxt);
			}
			return result;
		}
		throw new JsonMappingException(parser, "'name' required as second element");
	}
}