/*##########################################################################
#
# SignLive! CC cloud suite bridge.
#
# This is the protocol / bootstrapping part for the JavaScript client.
# Additional code for concrete bridglets is provided in csmodule files.
#
#   o_
# in|tarsys  AG (c)
#
#
 */
var csBridge = (function (window, undefined) {

  const
    VERSION = '1.0';

  const
    API_PATH_AGENT = '/bridge/api/v1';

  const
    API_PATH_AGENT_QUERY = '?_contentType=application/json';

  const
    API_PATH_MEETING = 'api/v1/meeting';

  const
    API_PATH_GEARS = 'api/v1';

  const
    GETTER_PREFIX = 'get';

  const
    METHOD_CREATE_BRIDGLET = 'createBridglet';

  const
    METHOD_DISPOSE = 'dispose';

  const
    METHOD_FETCH_MESSAGE = 'fetchMessage';

  const
    METHOD_ENTER = 'enter';

  const
    METHOD_OK = 'ok';

  const
    METHOD_CANCEL = 'cancel';

  const
    METHOD_FAIL = 'fail';

  const
    METHOD_CONNECT = 'connect';

  const
    XHR_STATUS_NOT_SET = 0;

  const
    XHR_START_OK = 200;

  const
    XHR_END_OK = 300;

  const
    MSG_TYPE_EVENT = 'event';

  const
    MSG_TYPE_CLOSE = 'close';

  const
    MSG_TYPE_PROCEED = 'proceed';

  const
    MSG_TYPE_ERROR = 'error';

  const
    MSG_TYPE_FETCH = 'fetch';

  const
    ERR_BRIDGE_UNEXPECTED_ERROR = 'Bridge.UnexpectedError';

  const
    ERR_BRIDGE_NOT_SUPPORTED = 'Bridglet.NotSupported';

  const
    ERR_BRIDGE_NOT_CONNECTED = 'Bridge.NotConnected';

  const
    ERR_BRIDGE_ALREADY_CONNECTED = 'Bridge.AlreadyConnected';

  const
    ERR_BRIDGE_DISPOSED = 'Bridge.Disposed';

  const
    ERR_BRIDGE_CONNECT_TIMEOUT = 'Bridge.ConnectTimeout';

  const
    ERR_BRIDGE_NOAGENT = 'Bridge.NoAgent';

  const
    ERR_BRIDGLET_UNKNOWN_FACTORY = 'Bridglet.UnknownFactory';

  const
    ERR_BRIDGLET_UNKNOWN_MESSAGE_TYPE = 'Bridglet.UnknownMessageType';

  const
    ERR_REMOTECALL_NO_CONNECTION = 'RemoteCall.NoConnection';

  const
    ERR_REMOTECALL_DATA_ERROR = 'RemoteCall.DataError';

  const
    ERR_REMOTECALL_UNKNOWN_ERROR = 'RemoteCall.UnknownError';

  const
    EVT_UPDATE = 'update';

  const
    CONNECT_TYPE_CONNECT = 'connect';

  const
    CONNECT_TYPE_REDIRECT = 'redirect';

  const
    CONNECT_TYPE_ERROR = 'error';

  const
    TYPE_UNDEFINED = 'undefined';

  const
    TYPE_OBJECT = 'object';

  const
    TYPE_FUNCTION = 'function';

  const
    TYPE_STRING = 'string';

  const
    PROPERTY_WRAPPED = '_wrapped';

  const
    PROPERTY_RESULT_VALUE = 'resultValue';

  const
    PROPERTY_CLASSIFIER = 'classifier';

  const
    PROPERTY_MESSAGE = 'message';

  const
    PROPERTY_CODE = 'code';

  const
    PROPERTY_STRING = 'string';

  const
    PROPERTY_ERROR = '_error';

  const
    STATE_CREATED = 'Created';

  const
    STATE_CONNECTING = 'Connecting';

  const
    STATE_CONNECTED = 'Connected';

  const
    STATE_CONNECTION_FAILED = 'ConnectionFailed';

  const
    STATE_CONNECTION_LOST = 'ConnectionLost';

  const
    STATE_DISPOSED = 'Disposed';

  const
    POLL_INTERVAL = 1000;

  var
    DEFAULT_HOST = '127.0.0.1';

  var
    DEFAULT_PORT = 15930;

  /***************************************************************************
   * common functions
   *
   **************************************************************************/
  function _withTrailingSlash(path) {
    if (path.charAt(path.length - 1) != '/') {
      return path + '/';
    }
    return path;
  }

  function _createUID() {
    var S4 = function () {
      return (((1 + Math.random()) * 0x10000) | 0).toString(16).substring(1);
    };
    return (S4() + S4() + "-" + S4() + "-" + S4() + "-" + S4() + "-" + S4() + S4() + S4());
  }

  function _capitalize(string) {
    return string[0].toUpperCase() + string.slice(1);
  }

  function _createObject(bridge, source, proto, propertiesObject) {
    if (source == null) {
      return null;
    }
    var object = Object.create(proto, propertiesObject);
    object.bridge = bridge;
    if (typeof source == TYPE_STRING) {
      object.handle = source;
    } else {
      object.attributes = source;
    }
    return object;
  }

  /*
  we handle a structure like
  {
    name: 'eventname',
    args: {
      anykey: 'anyvalue'
    }
  }
  */
  function _triggerEvent(target, event) {
    var tempArray = target.callbacks[event.name];
    if (typeof tempArray === TYPE_UNDEFINED) {
      return;
    }
    tempArray.forEach(function (f) {
      f.call(target, event);
    });
  }

  function _triggerUpdate(target, property, oldValue, newValue) {
    _triggerEvent(target, {
      name: EVT_UPDATE,
      args: {
        property: property,
        oldValue: oldValue,
        newValue: newValue
      }
    });
  }

  function _set(target, property, value) {
    var oldState = target[property];
    target[property] = value;
    _triggerUpdate(target, property, oldState, value);
  }

  function _addEventListener(target, name, f) {
    var tempArray = target.callbacks[name];
    if (typeof tempArray === TYPE_UNDEFINED) {
      tempArray = [];
      target.callbacks[name] = tempArray;
    }
    tempArray.push(f);
  }

  function _removeEventListener(target, name, f) {
    var tempArray = target.callbacks[name];
    if (typeof tempArray === TYPE_UNDEFINED) {
      return;
    }
    tempArray = tempArray.filter(function (item) {
      if (item !== f) {
        return item;
      }
    });
    target.callbacks[name] = tempArray;
  }

  function EncodedError(code, message) {
    this.code = code;
    this.message = message;
  }

  function _toEncodedError(error) {
    if (error.code) {
      return error;
    }
    return new EncodedError(ERR_BRIDGE_UNEXPECTED_ERROR, error.message);
  }

  /***************************************************************************
   * RemoteObject
   *
   * All stub objects to a remote should inherit from this one
   *
   **************************************************************************/
  function /* abstract */RemoteObject(bridge) {
    this.bridge = bridge;
  }

  // a stub has either a handle to the remote..
  RemoteObject.prototype.handle = null;

  // ... or prefetched attributes
  RemoteObject.prototype.attributes = null;

  RemoteObject.prototype._invoke = function (name, args) {
    var that = this;
    args._receiver = that.handle;
    return _callLocal(that.bridge, name, args).then(undefined, function (e) {
      if (e.code === ERR_REMOTECALL_NO_CONNECTION) {
        _bridgeTransitionConnectionLost(that.bridge, e);
      }
      throw e;
    });
  };

  RemoteObject.prototype._wantValue = function (name, getterName) {
    if (this.attributes === null) {
      if (typeof getterName === TYPE_UNDEFINED) {
        getterName = GETTER_PREFIX + _capitalize(name);
      }
      return this._invoke(getterName, {});
    } else {
      return Promise.resolve(this.attributes[name])
    }
  };

  RemoteObject.prototype._ref = function () {
    if (this.attributes === null) {
      return this.handle;
    } else {
      return this.attributes.id;
    }
  };

  /***************************************************************************
   * Bridge
   *
   **************************************************************************/

  function _bridgeTestDisposed(bridge) {
    if (bridge.isDisposed()) {
      console.log("csb: test disposed, bridge disposed");
      throw new EncodedError(ERR_BRIDGE_DISPOSED, "bridge is disposed");
    }
  }

  function _bridgeTestStop(bridge) {
    _bridgeTestDisposed(bridge);
    if (bridge.urls.length <= 0) {
      console.log("csb: test stop, bridge no urls");
      var error = new EncodedError(ERR_BRIDGE_NOAGENT, "bridge no agent found");
      _bridgeTransitionConnectionFailed(bridge, error);
      throw error;
    }
  }

  function _bridgeDetectUrls(bridge) {
    var host = bridge.desc.host;
    if (host === undefined) {
      host = DEFAULT_HOST;
    }
    var protocol = bridge.desc.protocol;
    if (protocol === undefined) {
      if (host === '127.0.0.1') {
        protocol = 'http';
      } else {
        protocol = 'https'
      }
    }
    var portStart = bridge.desc.portStart;
    if (portStart === undefined) {
      portStart = bridge.desc.port;
    }
    if (portStart === undefined) {
      portStart = DEFAULT_PORT;
    }
    var portStop = bridge.desc.portStop;
    if (portStop === undefined) {
      portStop = portStart;
    }
    bridge.urls = [];
    var tempUrl;
    for (var port = portStart; port <= portStop; port++) {
      tempUrl = protocol + '://' + host + ':' + port + API_PATH_AGENT + API_PATH_AGENT_QUERY;
      bridge.urls.push(tempUrl);
    }
  }

  function _bridgeDetectAgentConnectUrls(bridge) {
    console.log("csb: connect try [%d]...", bridge.attempts++);
    return _bridgeDetectAgentConnectUrl(bridge, 0);
  }

  function _bridgeDetectAgentConnectUrl(bridge, index) {
    _bridgeTestDisposed(bridge);
    _bridgeTransitionConnecting(bridge);
    bridge.url = bridge.urls[index];
    return _callLocal(bridge, METHOD_CONNECT, bridge.desc).then(
      function (result) {
        _bridgeTestDisposed(bridge);
        if (typeof result !== TYPE_OBJECT) {
          return _bridgeDetectAgentConnectFailed(bridge, index, "connect unknown result", true);
        }
        if (result.type === CONNECT_TYPE_CONNECT) {
          if (result.version != bridge.lib.version) {
            return _bridgeDetectAgentConnectFailed(bridge, index, "connect version mismatch", true);
          }
          return _bridgeDetectAgentConnectSuccess(bridge, index, result);
        }
        if (result.type === CONNECT_TYPE_REDIRECT) {
          console.log("csb: connect redirect");
          result.url = result.url.replace("127.0.0.1", bridge.desc.host);
          result.url = result.url + API_PATH_AGENT_QUERY;
          bridge.urls = [result.url];
          bridge.desc.select = null;
          return _bridgeDetectAgentConnectUrl(bridge, 0);
        }
        if (result.type === CONNECT_TYPE_ERROR) {
          return _bridgeDetectAgentConnectFailed(bridge, index, "connect server error " + result.message, true);
        }
        return _bridgeDetectAgentConnectFailed(bridge, index, "connect unknown result type " + result.type, true);
      },
      function (reason) {
        _bridgeTestDisposed(bridge);
        return _bridgeDetectAgentConnectFailed(bridge, index, "connect failed", false);
      });
  }

  function _bridgeDetectAgentConnectFailed(bridge, index, msg, skip) {
    if (skip) {
      console.log("csb: " + msg + " to " + bridge.url + ", skip");
      bridge.urls.splice(index, 1);
      index--;
    } else {
      console.log("csb: " + msg + " to " + bridge.url);
    }
    index++;
    if (bridge.urls.length <= index) {
      var error = new EncodedError(ERR_BRIDGE_NOAGENT, "no agent found");
      _bridgeTransitionConnectionFailed(bridge, error);
      throw error;
    }
    return _bridgeDetectAgentConnectUrl(bridge, index);
  }

  function _bridgeDetectAgentConnectSuccess(bridge, index, result) {
    console.log("csb: connect success " + bridge.url + ", session " + result.session);
    bridge.session = result.session;
    _bridgeTransitionConnected(bridge);
    return bridge;
  }

  function _bridgeTransitionConnecting(bridge) {
    _set(bridge, 'state', STATE_CONNECTING);
  }

  function _bridgeTransitionConnected(bridge) {
    _set(bridge, 'state', STATE_CONNECTED);
  }

  function _bridgeTransitionDisposed(bridge) {
    _set(bridge, 'state', STATE_DISPOSED);
  }

  function _bridgeTransitionConnectionLost(bridge, e) {
    _set(bridge, 'state', STATE_CONNECTION_LOST);
  }

  function _bridgeTransitionConnectionFailed(bridge, e) {
    _set(bridge, 'state', STATE_CONNECTION_FAILED);
  }

  function _bridgeCreateJSONData(bridge, name, args) {
    if (typeof args === TYPE_UNDEFINED) {
      args = {};
    }
    args._name = name;
    if (bridge.session != null) {
      args._session = bridge.session;
    }
    return JSON.stringify(args);
  }

  function _bridgeSendBeacon(bridge, name, args) {
    if (navigator && navigator.sendBeacon) {
      navigator.sendBeacon(bridge.url, _bridgeCreateJSONData(bridge, name,
        args));
      return Promise.resolve({});
    } else {
      // try regular remoteCall and ignore the result; might not work
      return _callLocal(bridge, name, args);
    }
  }

  /*
  this is a general purpose ajax call routine
  */
  function _call(url, hdrContentType, hdrAccept, unmarshal, request) {
    var call = new Promise(function (resolve, reject) {
      var xhr = new XMLHttpRequest();
      var message = 'XMLHttpRequest: request ' + url;
      xhr.open('POST', url, true);
      xhr.setRequestHeader('Content-Type', hdrContentType);
      xhr.setRequestHeader('Accept', hdrAccept);
      xhr.onreadystatechange = function (error) {
        if (xhr.readyState == 4) {
          if (xhr.status >= XHR_START_OK && xhr.status < XHR_END_OK) {
            try {
              resolve(unmarshal(xhr));
            } catch (e) {
              var message = 'response invalid (' + e.message + ')';
              console.error("csb: %s", message);
              reject(new EncodedError(ERR_REMOTECALL_DATA_ERROR, message));
            }
          } else {
            var errorObject;
            if (xhr.status == XHR_STATUS_NOT_SET) {
              /*
               * lost connection or never had one in the first place
               */
              errorObject = new EncodedError(ERR_REMOTECALL_NO_CONNECTION, "connection cannot be established");
            } else {
              console.error("csb: xhr error, status %d, responseText %s", xhr.status, xhr.responseText);
              try {
                var tmpResult = JSON.parse(xhr.responseText);
                if (tmpResult.hasOwnProperty(PROPERTY_ERROR)) {
                  // unwrap
                  tmpResult = tmpResult._error;
                }
                errorObject = new EncodedError(tmpResult.code, tmpResult.message);
              } catch (e) {
                errorObject = new EncodedError(ERR_REMOTECALL_UNKNOWN_ERROR, xhr.responseText);
              }
            }
            reject(errorObject);
          }
        }
      };
      xhr.withCredentials = true;
      try {
        xhr.send(request);
      } catch (error) {
        var message = 'xhr send error (' + error.message + ')';
        console.error("csb: %s", message);
        reject(new EncodedError(ERR_REMOTECALL_NO_CONNECTION, message));
      }
    });
    return call;
  }

  function _callLocal(bridge, name, args) {
    var request = _bridgeCreateJSONData(bridge, name, args);
    var url = bridge.url;
    var unmarshal = function (xhr) {
      var result = JSON.parse(xhr.responseText);
      if (result.hasOwnProperty(PROPERTY_WRAPPED)) {
        return result[PROPERTY_WRAPPED];
      } else {
        return result;
      }
    };
    // work around https://bugzilla.mozilla.org/show_bug.cgi?id=1376310
    return _call(url, 'text/plain', 'application/json', unmarshal, request);
  }

  function _callMeeting(bridglet, name, args) {
    var url = _withTrailingSlash(bridglet.desc.args.hostUrl);
    url = url + API_PATH_MEETING + "/" + name;
    var request = JSON.stringify(args);
    var hdrContentType = 'application/json';
    var hdrAccept = 'application/json';
    var unmarshal = function (xhr) {
      if (xhr.status == 204 || xhr.responseText.trim() == "") {
        return null;
      }
      return JSON.parse(xhr.responseText);
    }
    return _call(url, hdrContentType, hdrAccept, unmarshal, request);
  }

  function _callGears(bridglet, name, args) {
    var url = _withTrailingSlash(bridglet.desc.args.cloudsuiteUrl);
    url = url + API_PATH_GEARS + '/bridge/' + name;
    var request = JSON.stringify(args);
    var hdrContentType = 'application/json';
    var hdrAccept = 'application/json';
    var unmarshal = function (xhr) {
      if (xhr.status == 204 || xhr.responseText.trim() == "") {
        return null;
      }
      return JSON.parse(xhr.responseText);
    }
    return _call(url, hdrContentType, hdrAccept, unmarshal, request);
  }

  function _callRemoteProcessor(bridglet, name, args) {
    var url = bridglet.desc.args.hostUrl;
    var request = '_name="' + name + '"';
    for (var prop in args) {
      request = request + '&' + prop + '="' + args[prop] + '"';
    }
    var hdrContentType = 'application/x-www-form-urlencoded';
    var hdrAccept = '*';
    var unmarshal = function (xhr) {
      return xhr.responseText;
    }
    return _call(url, hdrContentType, hdrAccept, unmarshal, request);
  }

  function Bridge(lib, desc) {
    this.lib = lib;
    this.desc = desc;
    this.id = 'brwsr-' + _createUID();
    this.url = null;
    this.session = null;
    this.callbacks = {};
    this.attempts = 1;
    this.state = STATE_CREATED;
    this.agentState = "existing";
    this.bridglets = [];
  }
  Bridge.prototype = Object.create(Object.prototype);
  Bridge.prototype.constructor = Bridge;

  Bridge.prototype.addEventListener = function (name, f) {
    return _addEventListener(this, name, f);
  };

  Bridge.prototype.removeEventListener = function (name, f) {
    return _removeEventListener(this, name, f);
  };

  Bridge.prototype.isDisposed = function () {
    return this.state === STATE_DISPOSED;
  };

  Bridge.prototype.isConnectionFailed = function () {
    return this.state === STATE_CONNECTION_FAILED;
  };

  Bridge.prototype.isConnectionLost = function () {
    return this.state === STATE_CONNECTION_LOST;
  };

  Bridge.prototype.isCreated = function () {
    return this.state === STATE_CREATED;
  };

  Bridge.prototype.isConnected = function () {
    return this.state === STATE_CONNECTED;
  };

  Bridge.prototype.getState = function () {
    return this.state;
  };

  Bridge.prototype.getAgentState = function () {
    return this.agentState;
  };

  Bridge.prototype.isConnecting = function () {
    return this.state === STATE_CONNECTING;
  };

  function BridgeViaServer(lib, desc) {
    Bridge.call(this, lib, desc);
  }
  BridgeViaServer.prototype = Object.create(Bridge.prototype);
  BridgeViaServer.prototype.constructor = BridgeViaServer;

  BridgeViaServer.prototype.connect = function () {
    _bridgeTestDisposed(this);
    if (this.isConnected() || this.isConnecting()) {
      return Promise.reject(new EncodedError(ERR_BRIDGE_ALREADY_CONNECTED, "bridge is already connected"));
    }
    _bridgeTransitionConnecting(this);
    _bridgeTransitionConnected(this);
    return Promise.resolve(this);
  }

  BridgeViaServer.prototype.connectRetry = function (duration, interval) {
    return Promise.resolve(this);
  }

  BridgeViaServer.prototype.createInstanceSpec = function (params) {
    var serializeParams = function (name, value) {
      if (typeof value === TYPE_OBJECT) {
        var result = "";
        for (var prop in value) {
          var newName;
          if (name) {
            newName = name + '.' + prop;
          } else {
            newName = prop;
          }
          result = result + serializeParams(newName, value[prop]);
        }
        return result;
      } else {
        return name + '=' + value + ';';
      }
    }
    return serializeParams(undefined, params);
  }

  BridgeViaServer.prototype.createBridglet = function (params) {
    if (!this.isConnected()) {
      return Promise.reject(new EncodedError(ERR_BRIDGE_NOT_CONNECTED, "bridge is not connected"));
    }
    var that = this;
    return new Promise(function (resolve, reject) {
      /*
       * for some reason browsers reload the client page when the location
       * change is not further deferred. Fortunately the promise chain
       * ends here, so we can just do "setTimeout"
       */
      setTimeout(function () {
        if (params.args.launch != false) {
          that.triggerSchemeHandler(params);
        } else {
          console.log("createBridglet");
          console.log(that.id);
          console.log(params);
        }

        var constructor;
        var bridglet;
        if (params.factory.indexOf('Remote') >= 0) {
          constructor = RemoteProcessorBridglet;
        } else if (params.factory.indexOf('Gears') >= 0) {
          constructor = GearsBridglet;
        } else if (params.factory.indexOf('Meeting') >= 0) {
          constructor = MeetingBridglet;
        } else {
          return reject(new EncodedError(ERR_BRIDGLET_UNKNOWN_FACTORY, "unsupported bridglet factory " + params.factory));
        }
        if (typeof csModule === TYPE_FUNCTION) {
          bridglet = csModule({
            RemoteObject: shared.RemoteObject,
            Bridglet: constructor,
            _createObject: shared._createObject
          }, that, params);
        } else {
          bridglet = new constructor(that, params);
        }
        that.bridglets.push(bridglet);
        resolve(bridglet);
      }, 1);
    });
  };

  BridgeViaServer.prototype.dispose = function () {
    if (this.isDisposed()) {
      return Promise.resolve({});
    }
    console.log("csb: dispose");
    this.agentState = "unknown";
    this.session = null;
    _bridgeTransitionDisposed(this);
    return Promise.resolve({});
  }

  function BridgeViaServerJnlp(lib, desc) {
    BridgeViaServer.call(this, lib, desc);
  }
  BridgeViaServerJnlp.prototype = Object.create(BridgeViaServer.prototype);
  BridgeViaServerJnlp.prototype.constructor = BridgeViaServerJnlp;

  BridgeViaServerJnlp.prototype.triggerSchemeHandler = function (params) {
    var jnlpfileWithParams = this.desc.jnlpfile;
    jnlpfileWithParams = jnlpfileWithParams + '?arg=-noautostart';
    jnlpfileWithParams = jnlpfileWithParams + '&arg=-enter';
    jnlpfileWithParams = jnlpfileWithParams + '&arg=' + encodeURIComponent(this.createInstanceSpec(params));
    csBridge.util._triggerSchemeHandler('jnlp:' + jnlpfileWithParams);
  }

  function BridgeViaServerCli(lib, desc) {
    BridgeViaServer.call(this, lib, desc);
  }
  BridgeViaServerCli.prototype = Object.create(BridgeViaServer.prototype);
  BridgeViaServerCli.prototype.constructor = BridgeViaServerCli;

  BridgeViaServerCli.prototype.triggerSchemeHandler = function (params) {
    var urlWithParams = 'csbridge:///app';
    urlWithParams = urlWithParams + '?noautostart=';
    urlWithParams = urlWithParams + '&enter=' + encodeURIComponent(this.createInstanceSpec(params));
    csBridge.util._triggerSchemeHandler(urlWithParams);
  }

  function BridgeViaAgent(lib, desc) {
    Bridge.call(this, lib, desc);
  }
  BridgeViaAgent.prototype = Object.create(Bridge.prototype);
  BridgeViaAgent.prototype.constructor = BridgeViaAgent;

  BridgeViaAgent.prototype.connect = function (connectFailCallback) {
    _bridgeTestDisposed(this);
    if (this.isConnected() || this.isConnecting()) {
      return Promise.reject(new EncodedError(ERR_BRIDGE_ALREADY_CONNECTED, "bridge already connected"));
    }
    _bridgeTestStop(this);
    var promise = _bridgeDetectAgentConnectUrls(this);
    if (typeof connectFailCallback === TYPE_FUNCTION) {
      var that = this;
      promise = promise.then(undefined, function (reason) {
        _bridgeTestStop(that);
        console.info("csb: connect delegate to launcher");
        return connectFailCallback(that);
      });
    }
    return promise;
  };

  BridgeViaAgent.prototype.connectRetry = function (duration, interval) {
    _bridgeTestDisposed(this);
    if (this.isConnected() || this.isConnecting()) {
      return Promise.reject(new EncodedError(ERR_BRIDGE_ALREADY_CONNECTED, "bridge already connected"));
    }
    if (interval === undefined) {
      interval = 2000;
    }
    var end = Date.now() + duration;
    _bridgeTestStop(this);
    var that = this;
    return new Promise(function (resolve, reject) {
      // we hope bridge agent is about to start and we wait patiently for
      // it to come up. disposing is up to the user in the meantime.
      var retry = function () {
        if ((duration != -1) && (Date.now() > end)) {
          var error = new EncodedError(ERR_BRIDGE_CONNECT_TIMEOUT, "bridge connect timeout");
          _bridgeTransitionConnectionFailed(that, error);
          reject(error);
        } else {
          _bridgeDetectAgentConnectUrls(that).then(
            function (bridge) {
              resolve(bridge);
            },
            function (innerReason) {
              try {
                _bridgeTestStop(that);
                setTimeout(retry, interval);
              } catch (e) {
                reject(_toEncodedError(e));
              }
            }
          );
        }
      };
      retry();
    });
  };

  BridgeViaAgent.prototype.createBridglet = function (params) {
    if (!this.isConnected()) {
      return Promise.reject(new EncodedError(ERR_BRIDGE_NOT_CONNECTED, "bridge is not connected"));
    }
    var that = this;
    return _callLocal(that, METHOD_CREATE_BRIDGLET, params).then(
      function (result) {
        var bridglet;
        if (typeof csModule === TYPE_FUNCTION) {
          bridglet = csModule(shared, that, params);
        } else {
          bridglet = new Bridglet(that, params);
        }
        bridglet.handle = result.bridglet;
        that.bridglets.push(bridglet);
        return bridglet;
      },
      function (e) {
        if (e.code === ERR_REMOTECALL_NO_CONNECTION) {
          _bridgeTransitionConnectionLost(that, e);
        }
        throw e;
      }
    );
  };

  BridgeViaAgent.prototype.dispose = function (fireAndForget) {
    if (this.isDisposed()) {
      return Promise.resolve({});
    }
    console.log("csb: dispose");
    var result;
    if (this.state == STATE_CONNECTED) {
      if (typeof fireAndForget === TYPE_UNDEFINED) {
        fireAndForget = false;
      }
      if (fireAndForget) {
        result = _bridgeSendBeacon(this, METHOD_DISPOSE);
      } else {
        result = _callLocal(this, METHOD_DISPOSE, {}).then(function () {
          console.info("csb: bridge disposed");
        });
      }
    } else {
      result = Promise.resolve({});
    }
    this.agentState = "unknown";
    this.session = null;
    _bridgeTransitionDisposed(this);
    return result;
  };

  /***************************************************************************
   * Bridglet
   *
   **************************************************************************/
  function Bridglet(bridge, desc) {
    RemoteObject.call(this, bridge);
    this.desc = desc;
    this.connected = false;
    this.callbacks = {};
  }
  Bridglet.prototype = Object.create(RemoteObject.prototype);
  Bridglet.prototype.constructor = Bridglet;

  /***************************************************************************
   * message loop handling
   *
   **************************************************************************/
  function _processMessage(bridglet, message) {
    if (typeof message === TYPE_OBJECT) {
      var result;
      if (message.type === MSG_TYPE_EVENT) {
        result = _processEvent(bridglet, message);
      } else if (message.type === MSG_TYPE_CLOSE) {
        result = _processClose(bridglet, message);
      } else if (message.type === MSG_TYPE_PROCEED) {
        result = _processProceed(bridglet, message);
      } else if (message.type === MSG_TYPE_ERROR) {
        result = _processError(bridglet, message);
      } else {
        throw new EncodedError(ERR_BRIDGLET_UNKNOWN_MESSAGE_TYPE, "unexpected message encountered");
      }
      if (result != null) {
        result.type = MSG_TYPE_FETCH;
      }
      return result;
    }
    throw new EncodedError(ERR_BRIDGLET_UNKNOWN_MESSAGE_TYPE, "unexpected message encountered");
  }

  function _processEvent(bridglet, message) {
    /*
     * for "known" events we will extract and if necessary transform the
     * arguments and callback will be executed with the extracted arg. All
     * other callbacks will be executed with the event itself as parameter
     */
    var name = message.name;
    var event = message;
    if (name == null || name == undefined) {
      // legacy format
      name = message.event.type;
      event = {
        name: name,
        args: message.event
      }
    }
    var func = knownEvents[name];
    if (typeof func === TYPE_FUNCTION) {
      func(bridglet, event);
    } else {
      _triggerEvent(bridglet, event);
    }
    return {};
  }

  function _processClose(bridglet, message) {
    console.log("csb: message loop close");
    return null;
  }

  function _processProceed(bridglet, message) {
    return {};
  }

  function _processError(bridglet, message) {
    console.error("csb: message loop error " + message.error);
    return null;
  }

  function _messageLoop(bridglet) {
    var process = function (message) {
      if (bridglet.bridge.isDisposed()) {
        // termination because whole bridge was taken down
        // don't even handle anymore
        return undefined;
      }
      var response;
      try {
        response = _processMessage(bridglet, message);
      } catch (err) {
        console.error(err.message);
        // don't terminate loop
        response = {
          type: MSG_TYPE_FETCH
        };
      }
      if (response == null) {
        // "normal" termination in response to close message from server
        return undefined;
      }
      if (bridglet.bridge.isDisposed()) {
        // bridge was taken down while processing message
        return undefined;
      }
      bridglet._invoke(METHOD_FETCH_MESSAGE, response).then(process);
    };
    var args = {
      type: MSG_TYPE_FETCH,
      version: bridglet.bridge.lib.version
    };
    bridglet._invoke(METHOD_FETCH_MESSAGE, args).then(process);
  }

  Bridglet.prototype.enter = function () {
    var that = this;
    var connect = that._invoke(METHOD_ENTER, {}).then(function () {
      that.connected = true;
      // start message loop only after succesful connect
      _messageLoop(that);
      return Promise.resolve({});
    });
    return connect;
  };

  Bridglet.prototype.fail = function () {
    return this._invoke(METHOD_FAIL, {});
  };

  Bridglet.prototype.ok = function () {
    return this._invoke(METHOD_OK, {});
  };

  Bridglet.prototype.cancel = function (interrupt) {
    return this._invoke(METHOD_CANCEL, {
      'interrupt': !!interrupt
    });
  };

  Bridglet.prototype.dispose = function (interrupt) {
    this.connected = false;
    return Promise.resolve({});
  };

  Bridglet.prototype.wantResultValue = function () {
    return this._wantValue(PROPERTY_RESULT_VALUE);
  };

  Bridglet.prototype.addEventListener = function (name, f) {
    return _addEventListener(this, name, f);
  };

  Bridglet.prototype.removeEventListener = function (name, f) {
    return _removeEventListener(this, name, f);
  };

  function MeetingBridglet(bridge, desc) {
    Bridglet.call(this, bridge);
    this.desc = desc;
  }
  MeetingBridglet.prototype = Object.create(Bridglet.prototype);
  MeetingBridglet.prototype.constructor = MeetingBridglet;

  MeetingBridglet.prototype.enter = function () {
    var that = this;
    var messageLoopError = function (error) {
      console.debug("%s message loop error %o", that.id, error);
      var event = {
        name: "error",
        args: {
          error: error
        }
      };
      _triggerEvent(that, event);
    };
    var messageLoop = function (message) {
      if (!that.connected || that.bridge.isDisposed()) {
        console.debug("%s message loop terminated", that.id);
        return {};
      }
      // do *not* return promise
      that._fetchMessage().then(
        function (message) {
          try {
            that._processMessage(message);
            messageLoop();
          } catch (e) {
            messageLoopError(e);
          }
        },
        messageLoopError
      );
    };
    // we create a promise on the connection step and mark as connected on success
    // in addition we start the message loop concurrently...
    return this._connect().then(function () {
      that.connected = true;
      messageLoop();
      return {};
    });
  };

  MeetingBridglet.prototype.fail = function (code, message) {
    var args = {
      'meetingInfo': {
        'id': this.desc.args.state
      },
      'participantInfo': {
        'id': this.bridge.id
      },
      'name': 'postEvent',
      'args': {
        'name': 'fail',
        'error': {
          'code': code,
          'message': message
        }
      }
    };
    return _callMeeting(this, "call", args);
  };

  MeetingBridglet.prototype.cancel = function (interrupt) {
    var args = {
      'meetingInfo': {
        'id': this.desc.args.state
      },
      'participantInfo': {
        'id': this.bridge.id
      },
      'name': 'postEvent',
      'args': {
        'name': 'cancel'
      }
    };
    return _callMeeting(this, "call", args);
  };

  MeetingBridglet.prototype.ok = function () {
    var args = {
      'meetingInfo': {
        'id': this.desc.args.state
      },
      'participantInfo': {
        'id': this.bridge.id
      },
      'name': 'postEvent',
      'args': {
        'name': 'ok'
      }
    };
    return _callMeeting(this, "call", args);
  };

  MeetingBridglet.prototype.dispose = function (interrupt) {
    return this._disconnect();
  };

  MeetingBridglet.prototype._fetchMessage = function () {
    var args = {
      'meetingInfo': {
        'id': this.desc.args.state
      },
      'participantInfo': {
        'id': this.bridge.id
      }
    };
    return _callMeeting(this, "fetchMessage", args);
  };

  MeetingBridglet.prototype._connect = function () {
    var args = {
      'meetingInfo': {
        'id': this.desc.args.state
      },
      'participantInfo': {
        'id': this.bridge.id
      }
    };
    return _callMeeting(this, "connect", args);
  };

  MeetingBridglet.prototype._disconnect = function () {
    if (this.connected) {
      this.connected = false;
      var args = {
        'meetingInfo': {
          'id': this.desc.args.state
        },
        'participantInfo': {
          'id': this.bridge.id
        }
      };
      return _callMeeting(this, "disconnect", args);
    }
    return Promise.resolve({});
  };

  MeetingBridglet.prototype._processMessage = function (message) {
    if (message.type === MSG_TYPE_CLOSE) {
      this.connected = false;
      // already disconnected on server side, quit loop
    } else if (message.type === MSG_TYPE_PROCEED) {
    } else if (message.type === MSG_TYPE_ERROR) {
      throw new EncodedError(message.error.code, message.error.message);
    } else if (message.type === MSG_TYPE_EVENT) {
      try {
        _triggerEvent(this, message);
      } catch (e) {
        console.info("message loop event handler error", e);
      }
    } else {
      throw new EncodedError(ERR_BRIDGLET_UNKNOWN_MESSAGE_TYPE, "unexpected message encountered");
    }
  }

  MeetingBridglet.prototype._wantValue = function () {
    return Promise.resolve({});
  };

  function GearsBridglet(bridge, desc) {
    Bridglet.call(this, bridge);
    this.desc = desc;
  }
  GearsBridglet.prototype = Object.create(Bridglet.prototype);
  GearsBridglet.prototype.constructor = GearsBridglet;

  GearsBridglet.prototype.enter = function () {
    var that = this;
    let handler = new ConversationHandler();
    handler.urlAcknowledge = _withTrailingSlash(this.desc.args.cloudsuiteUrl);
    handler.urlAcknowledge = handler.urlAcknowledge + API_PATH_GEARS + '/flow/conversation/acknowledge';
    ConversationHandler.active = handler;
    handler.resumeSnapshot({
      "conversation": this.desc.args.conversation,
      "stage": {
        "id": 1
      }
    }).then(
      function (result) {
        _triggerEvent(that, {
          name: 'ok'
        });
      },
      function (error) {
        if (error.code === ERR_CANCEL_STAGE) {
          _triggerEvent(that, {
            name: 'cancel'
          });
        } else if (error.code.indexOf(ERR_ERROR_STAGE) === 0) {
          _triggerEvent(that, {
            name: 'fail',
            args: {
              error: error
            }
          });
        } else {
          _triggerEvent(that, {
            name: 'fail',
            args: {
              error: error
            }
          });
        }
      });
    return Promise.resolve({});
  };

  GearsBridglet.prototype.fail = function (interrupt) {
    return _callGears(this, 'outcome', {
      'conversation': this.desc.args.conversation,
      'state': 'fail'
    })
  };

  GearsBridglet.prototype.cancel = function (interrupt) {
    return _callGears(this, 'outcome', {
      'conversation': this.desc.args.conversation,
      'state': 'cancel'
    })
  };

  GearsBridglet.prototype.ok = function () {
    return _callGears(this, 'outcome', {
      'conversation': this.desc.args.conversation,
      'state': 'ok'
    })
  };

  GearsBridglet.prototype._wantValue = function () {
    return Promise.resolve({});
  };

  function RemoteProcessorBridglet(bridge, desc) {
    Bridglet.call(this, bridge);
    this.desc = desc;
  }
  RemoteProcessorBridglet.prototype = Object.create(Bridglet.prototype);
  RemoteProcessorBridglet.prototype.constructor = RemoteProcessorBridglet;

  RemoteProcessorBridglet.prototype.enter = function () {
    var that = this;
    var args = {
      jobId: this.desc.args.jobId
    };
    var messageLoop = function (message) {
      if (that.bridge.isDisposed()) {
        // termination because whole bridge was taken down
        // don't even handle anymore
        return undefined;
      }
      if (message === '"ok"') {
        try {
          _triggerEvent(that, {
            name: 'ok'
          });
        } finally {
          return;
        }
      } else if (message === '"failed"') {
        try {
          _triggerEvent(that, {
            name: 'fail',
            args: {
              error: {
                code: "unknown",
                message: "unknown reason"
              }
            }
          });
        } finally {
          return;
        }
      } else if (message === '"canceled"') {
        try {
          _triggerEvent(that, {
            name: 'cancel'
          });
        } finally {
          return;
        }
      }
      if (that.bridge.isDisposed()) {
        // bridge was taken down while processing message
        return undefined;
      }
      setTimeout(function () {
        _callRemoteProcessor(that, 'getJobState', args).then(messageLoop, errorHandler);
      }, POLL_INTERVAL);
    };
    var errorHandler = function () {
      // stop loop but otherwise ignore
    };
    _callRemoteProcessor(this, 'getJobState', args).then(messageLoop, errorHandler);
    return Promise.resolve({});
  };

  RemoteProcessorBridglet.prototype.fail = function (interrupt) {
    return Promise.reject(new EncodedError(ERR_BRIDGE_NOT_SUPPORTED, "unsupported function for remote processor"));
  };

  RemoteProcessorBridglet.prototype.cancel = function (interrupt) {
    return Promise.reject(new EncodedError(ERR_BRIDGE_NOT_SUPPORTED, "unsupported function for remote processor"));
  };

  RemoteProcessorBridglet.prototype.ok = function () {
    return Promise.reject(new EncodedError(ERR_BRIDGE_NOT_SUPPORTED, "unsupported function for remote processor"));
  };

  RemoteProcessorBridglet.prototype._wantValue = function () {
    return Promise.resolve({});
  };

  /***************************************************************************
   * specific events
   **************************************************************************/

  function Activity(bridge) {
    RemoteObject.call(this, bridge);
  }
  Activity.prototype = Object.create(RemoteObject.prototype);
  Activity.prototype.constructor = Activity;

  Activity.prototype.ok = function () {
    return this._invoke(METHOD_OK, {});
  }

  Activity.prototype.wantClassifier = function () {
    return this._wantValue(PROPERTY_CLASSIFIER);
  }

  Activity.prototype.wantMessage = function () {
    var that = this;
    return this._wantValue(PROPERTY_MESSAGE).then(function (message) {
      return _createMessage(that.bridge, message);
    });
  }

  function _createActivity(bridge, source) {
    return _createObject(bridge, source, Activity.prototype);
  }

  function Message(bridge) {
    RemoteObject.call(this, bridge);
  }
  Message.prototype = Object.create(RemoteObject.prototype);
  Message.prototype.constructor = Message;

  Message.prototype.wantCode = function () {
    return this._wantValue(PROPERTY_CODE);
  }

  Message.prototype.wantString = function () {
    return this._wantValue(PROPERTY_STRING);
  }

  function _createMessage(bridge, source) {
    return _createObject(bridge, source, Message.prototype);
  }

  var knownEvents = {};
  var activityEvent = function (target, event) {
    var tempArray = target.callbacks[event.name];
    if (typeof tempArray === TYPE_UNDEFINED) {
      return;
    }
    var activity = _createActivity(target.bridge, event.args.activity);
    tempArray.forEach(function (f) {
      f.call(target, activity);
    });
  }

  knownEvents.activityChanged = activityEvent;
  knownEvents.activityEnter = activityEvent;
  knownEvents.activityFailed = activityEvent;
  knownEvents.activityFinally = activityEvent;
  knownEvents.activityFinished = activityEvent;

  /***************************************************************************
   *
   **************************************************************************/

  var shared = {
    RemoteObject: RemoteObject,
    Bridglet: Bridglet,
    _createObject: _createObject
  };

  /***************************************************************************
   * the cloud suite bridge library module object
   *
   **************************************************************************/

  var tempCSBridge = {};

  tempCSBridge.util = {};

  tempCSBridge.lib = {
    version: VERSION
  };

  tempCSBridge.lib.createBridge = function (params) {
    var context = {
      // documentbase : window.location.origin
    };
    params.context = context;
    var bridge;
    if (params.mode === 'cli-csbridge') {
      /*
       * calling the csbridge protocol handler along with arguments
       */
      bridge = new BridgeViaServerCli(tempCSBridge.lib, params);
    } else if (typeof params.jnlpfile === TYPE_STRING) {
      /*
       * calling the jnlp protocol handler along with arguments
       */
      bridge = new BridgeViaServerJnlp(tempCSBridge.lib, params);
    } else {
      /*
       * calling the preloaded bridge web service
       */
      bridge = new BridgeViaAgent(tempCSBridge.lib, params);
      _bridgeDetectUrls(bridge);
    }
    return bridge;
  };

  tempCSBridge.lib.dispose = function (fireAndForget) {
    // for future use
    return Promise.resolve({});
  };

  /***************************************************************************
   * utility code
   *
   **************************************************************************/
  tempCSBridge.util.launchCSBridge = function (bridge) {
    console.info("csb: launch Auto ('csbridge' scheme handler) ...");
    bridge.agentState = "launchAuto";
    _triggerUpdate(bridge);
    this._triggerSchemeHandler('csbridge:///app');
    // can't detect outcome; assume success
    return Promise.resolve("launchAutoSuccess");
  }

  tempCSBridge.util.launchJNLP = function (bridge, args) {
    console.info("csb: launch Auto (Web Start plugin or 'jnlp' scheme handler) ...");
    bridge.agentState = "launchAuto";
    _triggerUpdate(bridge);
    var jnlpfile = args.jnlpfile;
    if (typeof dtjava === 'undefined') {
      this._triggerSchemeHandler('jnlp:' + jnlpfile);
      // can't detect outcome; assume success
      return Promise.resolve("launchAutoSuccess");
    }
    var app = new dtjava.App(jnlpfile, {
      id: 'cloudsuite-bridge',
      params: {},
      toolkit: "swing"
    });
    var platform = new dtjava.Platform({
      jvm: "1.8",
      jvmargs: "-Dbridge.fromJnlpFile=OK"
    });
    var promise = new Promise(function (resolve, reject) {
      var error = false;
      dtjava.launch(app, platform, {
        onDeployError: function (app, evt) {
          console.log("csb: JNLP deploy error " + evt);
          error = true;
        },
        onInstallNeeded: function (app, platform, cb, autoInstall, needRelaunch, launchFunction) {
          console.log("csb: JNLP install needed");
          /*
           * we cannot find out if Java isn't installed at all or if
           * the browser just doesn't use the plugin. Maybe we can
           * offer the user elsewhere to take them to the Java
           * download page.
           */
          // launch without prompting
          launchFunction();
        },
        onRuntimeError: function (e) {
          console.log("csb: JNLP runtime error " + e);
          error = true;
        }
      });
      // launch is a synchronous function
      if (error) {
        console.log("csb: launch auto error");
        bridge.agentState = "launchManual";
        _triggerUpdate(bridge);
        reject(new EncodedError(ERR_BRIDGE_NOAGENT, "launchAutoError"));
      } else {
        console.log("csb: launch auto success");
        bridge.agentState = "launchAutoSuccess";
        _triggerUpdate(bridge);
        resolve("launchAutoSuccess");
      }
    });
    return promise;
  };

  tempCSBridge.util._triggerSchemeHandler = function (uri) {
    /*
     * changing the document location will trigger onbeforeunload handlers.
     * We don't want that.
     */
    var previousOnbeforeunload = window.onbeforeunload;
    var emptyFunction = function () {
    };
    window.onbeforeunload = emptyFunction;
    document.location = uri;
    /*
     * location change will be executed asynchronously and there isn't a
     * notification when it has completed -> setTimeout is the only option
     * to put the handler back
     */
    setTimeout(function () {
      if (window.onbeforeunload === emptyFunction) {
        window.onbeforeunload = previousOnbeforeunload;
      }
    }, 300);
  };

  return tempCSBridge;

})(window);
