package de.intarsys.cloudsuite.gears.core.client;

import java.lang.reflect.Constructor;
import java.util.Map;

import de.intarsys.conversation.service.client.api.ConversationalResponse;
import de.intarsys.conversation.service.client.api.DtoConversationSnapshot;
import de.intarsys.conversation.service.client.api.DtoHttpRedirectStage;
import de.intarsys.conversation.service.client.api.DtoReplyStage;
import de.intarsys.conversation.service.client.api.DtoResultStage;
import de.intarsys.conversation.service.client.api.RequestAcknowledge;
import de.intarsys.conversation.service.client.api.RequestCancel;
import de.intarsys.conversation.service.client.api.ResponseAcknowledge;
import de.intarsys.conversation.service.client.api.ResponseCancel;
import de.intarsys.conversation.service.component.api.Outcome;
import de.intarsys.conversation.service.component.api.RequestResume;
import de.intarsys.conversation.service.component.api.ResponseResume;
import jakarta.ws.rs.core.Response;

/**
 * This is a conversational stub (which may have a page representation in case of redirect).
 */
public abstract class ConversationStub extends PageStub {

	private static final String OP_CANCEL = "cancel";

	private static final String OP_ACKNOWLEDGE = "acknowledge";

	private static final String PATH_ACKNOWLEDGE = "/api/v1/flow/conversation/acknowledge";

	private static final String PATH_CANCEL = "/api/v1/flow/conversation/cancel";

	private static final String PATH_RESUME = "/api/v1/flow/conversation/resume";

	protected static String createUrl(ConversationalResponse response) {
		DtoReplyStage stage = response.getSnapshot().getReplyStage();
		if (stage instanceof DtoHttpRedirectStage) {
			return ((DtoHttpRedirectStage) stage).getUrl();
		}
		return null;
	}

	private final String handle;

	private DtoReplyStage replyStage;

	private Map<String, String> links;

	protected ConversationStub(Protocol protocol, ConversationalResponse response)
			throws GearsServiceException {
		this(protocol, createUrl(response), response.getSnapshot().getConversationHandle());
		handleSnapshot(response.getSnapshot());
	}

	protected ConversationStub(Protocol protocol, String rootUrl, String handle) {
		super(protocol, rootUrl);
		this.handle = handle;
	}

	public ResponseAcknowledge acknowledge() throws GearsServiceException {
		return acknowledge(null);
	}

	public ResponseAcknowledge acknowledge(Object value) throws GearsServiceException {
		RequestAcknowledge reqAcknowledge = new RequestAcknowledge(getHandle());
		reqAcknowledge.setValue(value);
		ResponseAcknowledge ack = getProtocol().postJson(
				getLink(OP_ACKNOWLEDGE, ConversationStub.PATH_ACKNOWLEDGE),
				null /* headers */,
				createParameters(),
				reqAcknowledge,
				ResponseAcknowledge.class);
		handleSnapshot(ack.getSnapshot());
		return ack;
	}

	public ResponseCancel cancel() throws GearsServiceException {
		RequestCancel reqCancel = new RequestCancel(getHandle());
		return getProtocol().postJson(
				getLink(OP_CANCEL, ConversationStub.PATH_CANCEL),
				null /* headers */,
				createParameters(),
				reqCancel,
				ResponseCancel.class);
	}

	public <S extends CommonStub> S followRedirect(Class<S> target) throws GearsServiceException {
		try {
			DtoHttpRedirectStage stage = (DtoHttpRedirectStage) getReplyStage();
			Constructor<S> constructor = target.getConstructor(Protocol.class, String.class);
			S newStub = constructor.newInstance(getProtocol(), stage.getUrl());
			return newStub;
		} catch (Exception e) {
			throw new GearsServiceException(e);
		}

	}

	public String getHandle() {
		return handle;
	}

	protected String getLink(String operation, String defaultValue) {
		if (links == null) {
			return defaultValue;
		}
		String link = links.get(operation);
		if (link != null) {
			return link;
		}
		return defaultValue;
	}

	public DtoReplyStage getReplyStage() {
		return replyStage;
	}

	protected DtoHttpRedirectStage getReplyStageHttpRedirect() {
		return (DtoHttpRedirectStage) replyStage;
	}

	protected DtoResultStage getReplyStageResult() {
		return (DtoResultStage) replyStage;
	}

	protected void handleSnapshot(DtoConversationSnapshot snapshot) {
		setReplyStage(snapshot.getReplyStage());
		setLinks(snapshot.getLinks());
	}

	public Response resumeGet(Outcome outcome) throws GearsServiceException {
		Map<String, String> params = createParameters();
		params.put("state", getHandle());
		if (outcome != null) {
			params.put("outcome", outcome.toString());
			setReplyStage(new DtoResultStage());
		}
		return getProtocol().get(
				ConversationStub.PATH_RESUME,
				null /* headers */,
				params,
				Response.class);
	}

	public ResponseResume resumePost(Outcome outcome) throws GearsServiceException {
		RequestResume reqResume = new RequestResume(getHandle(), outcome);
		setReplyStage(new DtoResultStage());
		return getProtocol().postJson(
				ConversationStub.PATH_RESUME,
				null /* headers */,
				createParameters(),
				reqResume,
				ResponseResume.class);
	}

	protected void setLinks(Map<String, String> links) {
		this.links = links;
	}

	protected void setReplyStage(DtoReplyStage replyStage) {
		this.replyStage = replyStage;
	}
}
