package de.intarsys.cloudsuite.gears.core.client;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.function.Supplier;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;

import de.intarsys.tools.functor.ArgTools;
import de.intarsys.tools.functor.Args;
import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.ssl.SslTools;

public class HttpConfiguration {
	public static class Builder {
		private HttpConfiguration template = new HttpConfiguration();

		public HttpConfiguration build() {
			return new HttpConfiguration(template.sslContext, template.hostnameVerifier, template.connectTimeout,
					template.readTimeout, template.useProxy);
		}

		public Builder connectTimeout(long connectionTimeout) {
			template.connectTimeout = connectionTimeout;
			return this;
		}

		public Builder hostnameVerifier(HostnameVerifier hostnameVerifier) {
			template.hostnameVerifier = hostnameVerifier;
			return this;
		}

		public Builder readTimeout(long readTimeout) {
			template.readTimeout = readTimeout;
			return this;
		}

		public Builder sslContext(SSLContext sslContext) {
			template.sslContext = sslContext;
			return this;
		}

		public Builder useProxy() {
			template.useProxy = true;
			return this;
		}
	}

	public static final String ARG_SSL_CONTEXT = "sslContext";
	public static final String ARG_HOSTNAME_VERIFIER = "hostnameVerifier";
	public static final String ARG_CONNECT_TIMEOUT = "connectTimeout";
	public static final String ARG_READ_TIMEOUT = "readTimeout";

	public static final String ARG_USE_PROXY = "useProxy";
	private static final long DEFAULT_CONNECT_TIMEOUT = 0L;
	private static final long DEFAULT_READ_TIMEOUT = 0L;

	private static final boolean DEFAULT_USE_PROXY = false;

	public static Builder builder() {
		return new Builder();
	}

	public static HttpConfiguration createDefault() throws GeneralSecurityException, IOException {
		return new HttpConfiguration(
				SslTools.createSslContextDefault(),
				null,
				DEFAULT_CONNECT_TIMEOUT,
				DEFAULT_READ_TIMEOUT,
				DEFAULT_USE_PROXY);
	}

	public static HttpConfiguration from(IArgs args) {
		if (args == null) {
			return null;
		}
		SSLContext sslContext = SslTools.getSslContextOrDefault(args, ARG_SSL_CONTEXT);
		HostnameVerifier hostnameVerifier = ArgTools.getValue(args, ARG_HOSTNAME_VERIFIER, HostnameVerifier.class,
				null);
		int connectTimeout = ArgTools.getInt(args, ARG_CONNECT_TIMEOUT, 0);
		int readTimeout = ArgTools.getInt(args, ARG_CONNECT_TIMEOUT, 0);
		boolean useProxy = ArgTools.getBoolean(args, ARG_USE_PROXY, false);

		return new HttpConfiguration(sslContext, hostnameVerifier, connectTimeout, readTimeout, useProxy);
	}

	public static HttpConfiguration get(IArgs args, String path) {
		Object value = ArgTools.getPath(args, path);
		if (value instanceof HttpConfiguration httpConfiguration) {
			return httpConfiguration;
		}
		return from(ArgTools.toArgs(value));
	}

	public static HttpConfiguration get(IArgs args, String path, Supplier<HttpConfiguration> defaultValue) {
		Object value = ArgTools.getPath(args, path);
		if (value instanceof HttpConfiguration httpConfiguration) {
			return httpConfiguration;
		}
		HttpConfiguration result = from(ArgTools.toArgs(value));
		if (result == null) {
			result = defaultValue.get();
		}
		return result;
	}

	private SSLContext sslContext;
	private HostnameVerifier hostnameVerifier;
	private long connectTimeout;
	private long readTimeout;
	private boolean useProxy;

	public HttpConfiguration() {
		this(null, null, DEFAULT_CONNECT_TIMEOUT, DEFAULT_READ_TIMEOUT,
				DEFAULT_USE_PROXY);
	}

	public HttpConfiguration(SSLContext sslContext, HostnameVerifier hostnameVerifier, long connectTimeout,
			long readTimeout, boolean useProxy) {
		this.sslContext = sslContext;
		this.hostnameVerifier = hostnameVerifier;
		this.connectTimeout = connectTimeout;
		this.readTimeout = readTimeout;
		this.useProxy = useProxy;
	}

	public long getConnectTimeout() {
		return connectTimeout;
	}

	public HostnameVerifier getHostnameVerifier() {
		return hostnameVerifier;
	}

	public long getReadTimeout() {
		return readTimeout;
	}

	public SSLContext getSslContext() {
		return sslContext;
	}

	public boolean isUseProxy() {
		return useProxy;
	}

	public void setConnectTimeout(long connectTimeout) {
		this.connectTimeout = connectTimeout;
	}

	public void setHostnameVerifier(HostnameVerifier hostnameVerifier) {
		this.hostnameVerifier = hostnameVerifier;
	}

	public void setReadTimeout(long readTimeout) {
		this.readTimeout = readTimeout;
	}

	public void setSslContext(SSLContext sslContext) {
		this.sslContext = sslContext;
	}

	public void setUseProxy(boolean useProxy) {
		this.useProxy = useProxy;
	}

	public IArgs toArgs() {
		IArgs args = Args.create();
		args.put(ARG_SSL_CONTEXT, sslContext);
		args.put(ARG_HOSTNAME_VERIFIER, hostnameVerifier);
		args.put(ARG_CONNECT_TIMEOUT, connectTimeout);
		args.put(ARG_READ_TIMEOUT, readTimeout);
		args.put(ARG_USE_PROXY, useProxy);

		return args;
	}

	@Override
	public String toString() {
		return String.format("%s(sslContext=%s, hostnameVerifier=%s, connectTimeout=%d, readTimeout=%d, useProxy=%b)",
				getClass().getSimpleName(),
				sslContext == null ? "no" : "yes",
				hostnameVerifier == null ? "no" : "yes",
				connectTimeout,
				readTimeout,
				useProxy);
	}
}
