package de.intarsys.cloudsuite.gears.core.client;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;

import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.client.RequestEntityProcessing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationConfig;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.cfg.ContextAttributes;
import com.fasterxml.jackson.jakarta.rs.json.JacksonJsonProvider;

import de.intarsys.cloudsuite.gears.core.service.common.jackson.ITransportItemLocatorFactory;
import de.intarsys.cloudsuite.gears.core.service.common.jackson.TransportDocumentLocatorDeserializer;
import de.intarsys.tools.file.TempTools;
import de.intarsys.tools.jaxrs.exception.ResponseError;
import de.intarsys.tools.jaxrs.logging.LoggingFeature;
import de.intarsys.tools.locator.FileLocator;
import de.intarsys.tools.locator.ILocator;
import jakarta.ws.rs.ProcessingException;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.client.Invocation.Builder;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status.Family;

/**
 * The {@link Protocol} defines the wire format and request processing for a gears {@link CommonStub} .
 *
 * Processing is based on the JAXRS libraries.
 *
 */
public class Protocol {

	private static final Logger Log = LoggerFactory.getLogger(Protocol.class);

	private final URI endpoint;

	private final HttpConfiguration httpConfiguration;

	private final Client client;

	private String stickyFieldName = "sticky_id";

	private final List<IRequestProcessor> requestProcessors = new ArrayList<>();

	private final List<IResponseProcessor> responseProcessors = new ArrayList<>();

	public Protocol(Client client, URI endpoint) {
		this.client = client;
		this.endpoint = endpoint;
		this.httpConfiguration = new HttpConfiguration();
	}

	public Protocol(String endpoint) throws URISyntaxException {
		this(new URI(endpoint), new HttpConfiguration());
	}

	public Protocol(URI endpoint) {
		this(endpoint, new HttpConfiguration());
	}

	public Protocol(URI endpoint, HttpConfiguration httpConfiguration) {
		Objects.requireNonNull(endpoint);
		Objects.requireNonNull(httpConfiguration);
		this.httpConfiguration = httpConfiguration;
		this.endpoint = endpoint;
		this.client = createClient();
	}

	public Protocol(URL endpoint) throws URISyntaxException {
		this(endpoint.toURI(), new HttpConfiguration());
	}

	public Protocol(URL endpoint, HttpConfiguration httpConfiguration) throws URISyntaxException {
		this(endpoint.toURI(), httpConfiguration);
	}

	public void addRequestProcessor(IRequestProcessor requestProcessor) {
		this.requestProcessors.add(requestProcessor);
	}

	public void addResponseProcessor(IResponseProcessor responseProcessor) {
		this.responseProcessors.add(responseProcessor);
	}

	protected Client createClient() {
		LoggingFeature logging = new LoggingFeature(
				Log,
				Level.TRACE,
				LoggingFeature.Verbosity.PAYLOAD_TEXT,
				LoggingFeature.DEFAULT_MAX_ENTITY_SIZE);

		JacksonJsonProvider jacksonProvider = new JacksonJsonProvider(createObjectMapper());
		ClientConfig config = new ClientConfig() //
				.register(jacksonProvider) //
				// .register(MultiPartFeature.class) //
				.register(logging)
				.property(ClientProperties.FOLLOW_REDIRECTS, false);

		ClientBuilder builder = ClientBuilder
				.newBuilder()
				.withConfig(config)
				.connectTimeout(httpConfiguration.getConnectTimeout(), TimeUnit.MILLISECONDS)
				.readTimeout(httpConfiguration.getReadTimeout(), TimeUnit.MILLISECONDS);

		if (httpConfiguration.isUseProxy()) {
			String proxyHost = System.getProperty("http.proxyHost");
			if (proxyHost == null) {
				Log.error("Proxy enabled, but system property http.proxyHost is not set; continuing without proxy");
			} else {
				// TODO Filter http.nonProxyHosts
				String proxyScheme = proxyHost.startsWith("http")
						? ""
						: "http://";
				String proxyPort = System.getProperty("http.proxyPort");
				String proxyUri = proxyPort == null
						? proxyScheme + proxyHost
						: proxyScheme + proxyHost + ':' + proxyPort;

				String proxyUser = System.getProperty("http.proxyUser");
				String proxyPassword = System.getProperty("http.proxyPassword");

				builder.property(ClientProperties.PROXY_URI, proxyUri)
						.property(ClientProperties.PROXY_USERNAME, proxyUser)
						.property(ClientProperties.PROXY_PASSWORD, proxyPassword)
						// Ensure Content-Length header is added before requests are sent to the proxy
						.property(ClientProperties.REQUEST_ENTITY_PROCESSING, RequestEntityProcessing.BUFFERED);
			}
		}

		HostnameVerifier hostnameVerifier = httpConfiguration.getHostnameVerifier();
		if (hostnameVerifier != null) {
			builder.hostnameVerifier(hostnameVerifier);
		}

		SSLContext sslContext = httpConfiguration.getSslContext();
		if (sslContext != null) {
			builder.sslContext(sslContext);
		}
		return builder.build();
	}

	protected ObjectMapper createObjectMapper() {
		ObjectMapper mapper = new ObjectMapper();
		DeserializationConfig dc = mapper.getDeserializationConfig();
		ContextAttributes attrs = ContextAttributes.getEmpty();
		/*
		 * here we set up a streaming factory to the temp folder
		 */
		ITransportItemLocatorFactory factory = new ITransportItemLocatorFactory() {
			private int counter;

			@Override
			public ILocator createLocator(JsonParser parser, DeserializationContext ctxt) throws IOException {
				return new FileLocator(new File(TempTools.getTempDir(), "download-" + counter++ + ".bytes"));
			}
		};
		attrs = TransportDocumentLocatorDeserializer.setLocatorFactory(attrs, factory);
		dc = dc.with(attrs);
		mapper.setConfig(dc);
		return mapper;
	}

	public <RES> RES get(String path, Map<String, String> headers, Map<String, String> params,
			Class<RES> responseType)
			throws GearsServiceException {
		WebTarget target = target(path);
		if (params != null) {
			for (Map.Entry<String, String> entry : params.entrySet()) {
				target = target.queryParam(entry.getKey(), entry.getValue());
			}
		}
		Builder builder = target.request(MediaType.APPLICATION_JSON_TYPE);
		if (headers != null) {
			for (Map.Entry<String, String> entry : headers.entrySet()) {
				builder = builder.header(entry.getKey(), entry.getValue());
			}
		}
		for (IRequestProcessor processor : requestProcessors) {
			processor.process(builder);
		}
		try {
			Response response = builder.get();
			return handleResponse(response, responseType);
		} catch (ProcessingException e) {
			throw new GearsServiceException(e);
		} catch (GearsServiceException e) {
			throw e;
		}
	}

	public Client getClient() {
		return client;
	}

	/**
	 * This is the service root endpoint - all targets are expected relative to this address.
	 *
	 */
	public URI getEndpoint() {
		return endpoint;
	}

	public HttpConfiguration getHttpConfiguration() {
		return httpConfiguration;
	}

	public String getStickyFieldName() {
		return stickyFieldName;
	}

	protected <RES> RES handleResponse(Response response, Class<RES> responseType)
			throws GearsServiceException {
		try {
			for (IResponseProcessor processor : responseProcessors) {
				response = processor.process(response);
			}
			if (responseType == Response.class) {
				return (RES) response;
			}
			if (response.getStatusInfo().getFamily().equals(Family.SUCCESSFUL)) {
				return response.readEntity(responseType);
			} else {
				if (response.getStatusInfo().equals(Response.Status.UNAUTHORIZED)) {
					throw new GearsServiceException(response.getStatus(), "unauthorized", "authentication failed");
				} else if (MediaType.APPLICATION_JSON_TYPE.equals(response.getMediaType())) {
					ResponseError error = response.readEntity(ResponseError.class);
					if (error != null) {
						throw new GearsServiceException(response.getStatus(), error.getError().getCode(),
								error.getError().getMessage());
					}
				}
			}
			throw new GearsServiceException(response.getStatus(), "undefined", Optional.ofNullable(response
					.getStatusInfo().getReasonPhrase()).orElse("undefined"));
		} catch (GearsServiceException e) {
			throw e;
		} catch (Exception e) {
			throw new GearsServiceException(e);
		}
	}

	/**
	 * Launch a POST request to the sub-resource {@code path} using HTTP {@code headers}, URL query {@code parameters}
	 * and the payload {@code entity}.
	 * 
	 * The response is marshalled to a {@code responseType}.
	 * 
	 * @param <REQ>
	 *            The request entity type
	 * @param <RES>
	 *            The response entity type
	 * @param path
	 *            The sub resource of the protocol endpoint
	 * @param headers
	 *            Optional HTTP headers
	 * @param parameters
	 *            Optional query parameters
	 * @param entity
	 *            The payload
	 * @param responseType
	 *            The expected response type
	 * @return
	 * @throws GearsServiceException
	 */
	public <REQ, RES> RES postEntity(String path, Map<String, String> headers,
			Map<String, String> parameters,
			Entity<REQ> entity, Class<RES> responseType) throws GearsServiceException {
		WebTarget target = target(path);
		if (parameters != null) {
			for (Map.Entry<String, String> entry : parameters.entrySet()) {
				target = target.queryParam(entry.getKey(), entry.getValue());
			}
		}
		Builder builder = target.request(MediaType.WILDCARD_TYPE);
		if (headers != null) {
			for (Map.Entry<String, String> entry : headers.entrySet()) {
				builder = builder.header(entry.getKey(), entry.getValue());
			}
		}
		for (IRequestProcessor processor : requestProcessors) {
			processor.process(builder);
		}
		try {
			Response response = builder.post(entity);
			return handleResponse(response, responseType);
		} catch (ProcessingException e) {
			throw new GearsServiceException(e);
		} catch (GearsServiceException e) {
			throw e;
		}
	}

	/**
	 * Launch a POST request to the sub-resource {@code path} using HTTP {@code headers}, URL query {@code parameters}
	 * and the payload {@code request}. The request objects is marshalled to JSON.
	 * 
	 * The response is marshalled to a {@code responseType}.
	 * 
	 * @param <REQ>
	 *            The request entity type
	 * @param <RES>
	 *            The response entity type
	 * @param path
	 *            The sub resource of the protocol endpoint
	 * @param headers
	 *            Optional HTTP headers
	 * @param parameters
	 *            Optional query parameters
	 * @param entity
	 *            The payload
	 * @param responseType
	 *            The expected response type
	 * @return
	 * @throws GearsServiceException
	 */
	public <REQ, RES> RES postJson(String path, Map<String, String> headers,
			Map<String, String> parameters, REQ request,
			Class<RES> responseType)
			throws GearsServiceException {
		Entity<REQ> entity = Entity.json(request);
		return postEntity(path, headers, parameters, entity, responseType);
	}

	/**
	 * Launch a POST request to the sub-resource {@code path} using HTTP {@code headers}, URL query {@code parameters}
	 * and the payload {@code request}. The request objects is marshaled to JSON.
	 * 
	 * The response is marshaled to a {@code responseType}.
	 * 
	 * @param <REQ>
	 *            The request entity type
	 * @param <RES>
	 *            The response entity type
	 * @param path
	 *            The sub resource of the protocol endpoint
	 * @param headers
	 *            Optional HTTP headers
	 * @param request
	 *            The request object
	 * @param responseType
	 *            The expected response type
	 * @return
	 * @throws GearsServiceException
	 */
	public <REQ, RES> RES postJson(String path, Map<String, String> headers,
			REQ request,
			Class<RES> responseType)
			throws GearsServiceException {
		return postJson(path, headers, null, request, responseType);
	}

	/**
	 * Launch a POST request to the sub-resource {@code path} using the payload {@code request}.
	 * 
	 * The request objects is marshaled to JSON.
	 * 
	 * The response is marshaled to a {@code responseType}.
	 * 
	 * @param <REQ>
	 *            The request entity type
	 * @param <RES>
	 *            The response entity type
	 * @param path
	 *            The sub resource of the protocol endpoint
	 * @param request
	 *            The request object
	 * @param responseType
	 *            The expected response type
	 * @return
	 * @throws GearsServiceException
	 */
	public <REQ, RES> RES postJson(String path,
			REQ request,
			Class<RES> responseType)
			throws GearsServiceException {
		return postJson(path, null, null, request, responseType);
	}

	public boolean removeRequestProcessor(IRequestProcessor requestProcessor) {
		return this.requestProcessors.remove(requestProcessor);
	}

	public boolean removeResponseProcessor(IResponseProcessor responseProcessor) {
		return this.responseProcessors.remove(responseProcessor);
	}

	public void setStickyFieldName(String stickyFieldName) {
		this.stickyFieldName = stickyFieldName;
	}

	protected WebTarget target(String path) {
		URI uri = URI.create(path);
		if (uri.isAbsolute()) {
			return getClient().target(path);
		} else {
			return getClient().target(getEndpoint()).path(path);
		}
	}

}
