package de.intarsys.cloudsuite.gears.core.conversation;

import de.intarsys.cloudsuite.gears.core.client.ConversationStub;
import de.intarsys.cloudsuite.gears.core.client.GearsServiceException;
import de.intarsys.conversation.service.client.api.DtoCancelStage;
import de.intarsys.conversation.service.client.api.DtoErrorDetail;
import de.intarsys.conversation.service.client.api.DtoErrorStage;
import de.intarsys.conversation.service.client.api.DtoHttpRedirectStage;
import de.intarsys.conversation.service.client.api.DtoIdleStage;
import de.intarsys.conversation.service.client.api.DtoProcessingStage;
import de.intarsys.conversation.service.client.api.DtoReplyStage;
import de.intarsys.conversation.service.client.api.DtoRequestInputStage;
import de.intarsys.conversation.service.client.api.DtoResultStage;
import de.intarsys.tools.conversation.IConversation;
import de.intarsys.tools.conversation.IReplyStage;
import de.intarsys.tools.conversation.impl.Conversation;
import de.intarsys.tools.conversation.impl.HttpRedirectStage;
import de.intarsys.tools.conversation.impl.IdleStage;
import de.intarsys.tools.conversation.impl.ProcessingStage;
import de.intarsys.tools.conversation.impl.RequestInputStage;
import de.intarsys.tools.exception.ExceptionTools;

/**
 * Adapt the gears wire communication to an {@link IConversation} protocol.
 * 
 * For each {@link DtoReplyStage} an appropriate {@link IReplyStage} is published on the wrapped {@link IConversation}.
 * 
 * The client code is responsible for handling and driving the conversation. This implementation does not react in any
 * way on a state received by the stub, it only converts and forwards.
 * 
 */
public class ConversationStubConversationAdapter {

	private final IConversation conversation;

	public ConversationStubConversationAdapter() {
		this.conversation = new Conversation<>("ConversationStubConversationAdapter");
	}

	public ConversationStubConversationAdapter(IConversation conversation) {
		super();
		this.conversation = conversation;
	}

	protected void acknowledge(ConversationStub stub, Object value) {
		try {
			stub.acknowledge(value);
			handle(stub);
		} catch (Exception e) {
			localError(stub, e);
		}
	}

	protected void cleanup(ConversationStub stub) {
		if (!stub.getReplyStage().isFinal()) {
			try {
				// external cancel
				stub.cancel();
				// ...and conversation cleanup
				stub.acknowledge();
			} catch (GearsServiceException e) {
				//
			}
		}
	}

	protected void errorSchemeNotSupported(DtoReplyStage replyStage) {
		throw new UnsupportedOperationException("scheme '" + replyStage.getScheme() + "' not supported");
	}

	protected IReplyStage fromDto(DtoReplyStage dtoReplyStage) {
		if (dtoReplyStage instanceof DtoHttpRedirectStage dtoHttpRedirectStage) {
			return new HttpRedirectStage(dtoHttpRedirectStage.getUrl(), dtoHttpRedirectStage.isOutOfBand(), false);
		}
		if (dtoReplyStage instanceof DtoRequestInputStage dtoInputStage) {
			RequestInputStage inputStage = new RequestInputStage();
			inputStage.setType(dtoInputStage.getType());
			inputStage.setTitle(dtoInputStage.getTitle());
			inputStage.setMessage(dtoInputStage.getMessage());
			return inputStage;
		}
		if (dtoReplyStage instanceof DtoProcessingStage dtoProcessingStage) {
			ProcessingStage processingStage = new ProcessingStage();
			processingStage.setState(dtoProcessingStage.getState());
			processingStage.setMessage(dtoProcessingStage.getMessage());
			processingStage.setWork(dtoProcessingStage.getWork());
			processingStage.setWorked(dtoProcessingStage.getWorked());
			processingStage.setPollDelay(dtoProcessingStage.getPollDelay());
			return processingStage;
		}
		if (dtoReplyStage instanceof DtoIdleStage) {
			return new IdleStage();
		}
		errorSchemeNotSupported(dtoReplyStage);
		// unreachable code
		return null;
	}

	public IConversation getConversation() {
		return conversation;
	}

	public final void handle(ConversationStub stub) {
		try {
			DtoReplyStage replyStage = stub.getReplyStage();
			if (replyStage instanceof DtoCancelStage) {
				handleCancel(stub, replyStage);
			} else if (replyStage instanceof DtoErrorStage) {
				handleError(stub, replyStage);
			} else if (replyStage instanceof DtoResultStage) {
				handleResult(stub, replyStage);
			} else {
				handleForward(stub, replyStage);
			}
		} catch (RuntimeException e) {
			localError(stub, e);
		}
	}

	protected void handleCancel(ConversationStub stub, DtoReplyStage replyStage) {
		getConversation().cancel(true);
	}

	protected void handleError(ConversationStub stub, DtoReplyStage replyStage) {
		DtoErrorStage errorStage = (DtoErrorStage) replyStage;
		DtoErrorDetail error = errorStage.getError();
		GearsServiceException ex = new GearsServiceException(error.getCode(), error.getMessage());
		getConversation().completeExceptionally(ex);
	}

	protected void handleForward(ConversationStub stub, DtoReplyStage replyStage) {
		IReplyStage forwardStage = fromDto(replyStage);
		forwardStage.onAck((value) -> acknowledge(stub, value));
		getConversation().setReplyStage(forwardStage);
	}

	protected void handleResult(ConversationStub stub, DtoReplyStage replyStage) {
		DtoResultStage resultStage = (DtoResultStage) replyStage;
		getConversation().complete(resultStage.getResult());
	}

	/**
	 * Handle an error in the local side.
	 * 
	 * The protocol is cancelled and the local conversation completed with an exception.
	 * 
	 * @param stub
	 * @param e
	 */
	protected void localError(ConversationStub stub, Exception e) {
		cleanup(stub);
		getConversation().completeExceptionally(ExceptionTools.unwrap(e));
	}
}
