package de.intarsys.signlive.demo.api.http.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Base64;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HTTP;

import de.intarsys.tools.file.FileTools;
import de.intarsys.tools.json.Json;
import de.intarsys.tools.json.JsonArray;
import de.intarsys.tools.json.JsonObject;
import de.intarsys.tools.json.JsonReader;
import de.intarsys.tools.stream.StreamTools;

public class SignatureValidationClient {

	public static void main(String[] args) {
		try {
			new SignatureValidationClient().run(args);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	protected String getArg(String[] args, int index, String defaultValue) {
		if (args.length > index) {
			return args[index];
		}
		return defaultValue;
	}

	protected void processResponse(HttpResponse response, File reportFile) throws Exception {
		String contentType = null;
		Header[] headers = response.getHeaders(HTTP.CONTENT_TYPE);
		if (headers != null && headers.length > 0) {
			contentType = headers[0].getValue();
			if (contentType != null) {
				contentType = contentType.split(";")[0];
			}
		}
		if (contentType == null) {
			contentType = "application/octet-stream";
		}
		switch (contentType) {
		case "application/json":
			processResponseJson(response, reportFile);
			break;

		case "multipart/form-data":
			processResponseForm(response, reportFile);
			break;

		default:
			processResponseRaw(response, reportFile);
			break;
		}
	}

	protected void processResponseForm(HttpResponse response, File reportFile) throws Exception {
		HttpEntity resultEntity = response.getEntity();
		File formFile = new File(reportFile.getParentFile(), FileTools.getBaseName(reportFile) + ".multipart");
		StreamTools.copyStream(resultEntity.getContent(), true, new FileOutputStream(formFile), true);
		System.out.println("Result data written to: " + formFile.getAbsolutePath());
	}

	protected void processResponseJson(HttpResponse response, File reportFile) throws Exception {
		HttpEntity resultEntity = response.getEntity();
		JsonObject object = null;
		try (InputStream is = resultEntity.getContent()) {
			JsonReader reader = new JsonReader(is);
			object = reader.readJsonObject();
		}

		File jsonFile = new File(reportFile.getParentFile(), FileTools.getBaseName(reportFile) + ".json");
		FileTools.write(jsonFile, object.toString(3));
		System.out.println("Result data written to: " + jsonFile.getAbsolutePath());

		JsonObject report = (JsonObject) object.get("report");
		if (report != null) {
			String reportContent = (String) report.get("content");
			FileTools.write(reportFile, Base64.getDecoder().decode(reportContent));
			System.out.println("Validation report written to: " + reportFile.getAbsolutePath());
		}
	}

	protected void processResponseRaw(HttpResponse response, File reportFile) throws Exception {
		HttpEntity resultEntity = response.getEntity();
		InputStream is = null;
		OutputStream os = null;
		try {
			is = resultEntity.getContent();
			os = new FileOutputStream(reportFile);
			StreamTools.copyStream(is, false, os, false);
		} finally {
			StreamTools.close(is);
			StreamTools.close(os);
		}
		System.out.println("Validation report written to: " + reportFile.getAbsolutePath());
	}

	public void run(String[] args) throws Exception {
		String inName = getArg(args, 0, "../data/doc_signed.pdf");
		String outName = getArg(args, 1, System.getProperty("java.io.tmpdir") + "/data/doc_signed.report.pdf");
		String hostname = getArg(args, 2, "localhost");
		String servicename = getArg(args, 3, "/demo/SignatureValidationService");
		int port = Integer.parseInt(getArg(args, 4, "8080"));
		/*
		 * application/json
		 * multipart/form-data
		 * application/octet-stream
		 */
		String accept = getArg(args, 5, null);
		//
		URL url = new URL("http", hostname, port, servicename);
		HttpResponse response = null;
		try {
			// prepare HTTP call
			HttpClient client = HttpClientBuilder.create().build();
			HttpPost method = new HttpPost(url.toString());
			if (accept != null) {
				method.addHeader("Accept", accept);
			}

			JsonObject request = Json.createObject();

			FileInputStream filestream = new FileInputStream(inName);
			byte[] doc = StreamTools.toByteArray(filestream);
			String encodedDoc = Base64.getEncoder().encodeToString(doc);

			// add parameters
			JsonObject document = Json.createObject();
			document.put("name", inName);
			document.put("content", encodedDoc);
			request.put("document", document);

			JsonArray signatureContainers = Json.createArray();
			String signatureName = inName + ".p7s";
			if (new File(signatureName).isFile()) {
				FileInputStream signatureFilestream = new FileInputStream(signatureName);
				byte[] signatureDoc = StreamTools.toByteArray(signatureFilestream);
				String encodedSignatureDoc = Base64.getEncoder().encodeToString(signatureDoc);
				JsonObject signatureContainer = Json.createObject();
				signatureContainer.put("name", signatureName);
				signatureContainer.put("content", encodedSignatureDoc);
				signatureContainers.add(signatureContainer);
			}
			request.put("signatureContainers", signatureContainers);

			StringEntity entity = new StringEntity(request.toString(3), ContentType.APPLICATION_JSON);
			method.setEntity(entity);

			response = client.execute(method);
			if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
				// handle error
				return;
			}

			processResponse(response, new File(outName));
		} finally {
			if (response != null && response.getEntity() != null) {
				response.getEntity().consumeContent();
			}
		}
	}
}
