/*
 * CABAReT solutions
 * all rights reserved
 *
 */
package com.cabaret.demo.embedded.java;

import java.awt.Button;
import java.awt.FileDialog;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Label;
import java.awt.Panel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;

import javax.swing.BoxLayout;

import com.cabaret.api.java.JavaAPITools;
import com.cabaret.claptz.common.wizard.WizardTools;

import de.intarsys.claptz.launcher.SingletonContainerLauncher;
import de.intarsys.document.common.DocumentTools;
import de.intarsys.document.model.IDocument;
import de.intarsys.document.model.IDocumentType;
import de.intarsys.processor.document.model.IDocumentProcessor;
import de.intarsys.tools.event.Event;
import de.intarsys.tools.event.INotificationListener;
import de.intarsys.tools.event.StoppedEvent;
import de.intarsys.tools.functor.Args;
import de.intarsys.tools.functor.FunctorCancelledException;
import de.intarsys.tools.functor.FunctorTools;
import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.reflect.MethodException;

/**
 * This is an application of "hosted" mode. You have an application of your own
 * and want to include features (up to and including GUI) from a claptz
 * container. The "two" applications will run side by side.
 * <p>
 * The subclasses implement concrete embedding scenarios.
 */
abstract public class DemoHostedAWT {

	private IDocumentProcessor viewer;

	private Frame frame;

	private Label stateLabel;

	protected void createView() {
		frame = new Frame("My application");
		frame.setSize(450, 250);
		frame.addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				shutdownContainer();
				System.exit(0);
			}
		});
		frame.setLayout(new BoxLayout(frame, BoxLayout.Y_AXIS));
		Panel content = new Panel();
		content.setLayout(new FlowLayout());
		frame.add(content);
		Label label = new Label("application state:");
		frame.add(label);
		stateLabel = new Label("undefined");
		frame.add(stateLabel);
		Panel buttons = new Panel();
		buttons.setLayout(new BoxLayout(buttons, BoxLayout.X_AXIS));
		frame.add(buttons);
		//
		Button btnOpen = new Button("Open"); //$NON-NLS-1$
		btnOpen.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				onOpen();
			}
		});
		buttons.add(btnOpen);
		Button btnCreate = new Button("Create"); //$NON-NLS-1$
		btnCreate.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				onCreate();
			}
		});
		buttons.add(btnCreate);
		Button btnClose = new Button("Close"); //$NON-NLS-1$
		btnClose.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				onClose();
			}
		});
		buttons.add(btnClose);
		Button btnCall = new Button("Call"); //$NON-NLS-1$
		btnCall.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				onCall();
			}
		});
		buttons.add(btnCall);
		frame.setVisible(true);
	}

	protected IDocument load() throws IOException {
		// IDocument is a document abstraction
		// you can get the document implementation object - for example a jPod
		// PDDocument and do whatever you want...
		//
		// PDDocument pddoc = (PDDocument)idoc.getImpl();
		//
		FileDialog dialog = new FileDialog(frame, "Open"); //$NON-NLS-1$
		try {
			dialog.setVisible(true);
			String filename = dialog.getFile();
			if (filename != null) {
				return DocumentTools.load(dialog.getDirectory() + "/" + filename);
			}
			return null;
		} finally {
			dialog.dispose();
		}
	}

	protected void onCall() {
		try {
			setStateText("calling...");
			JavaAPITools.call("hello");
			setStateText("called");
		} catch (MethodException e) {
			setStateText("call failed");
			e.printStackTrace();
		}
	}

	protected void onClose() {
		if (viewer == null) {
			return;
		}
		viewer.stop();
		viewer = null;
	}

	protected void onCreate() {
		if (viewer != null) {
			viewer.stop();
		}
		IDocument idoc = null;
		try {
			Args creatorArgs = Args.create();
			creatorArgs.put(IDocumentType.ARG_DOCUMENTTYPE, "com.cabaret.document.pdf.PDFDocumentType");
			idoc = (IDocument) WizardTools.processProcessorFactoryId("DocumentCreatorFactory", creatorArgs);
			if (idoc == null) {
				return;
			}
			Args viewerArgs = Args.create();
			viewerArgs.put(IDocumentProcessor.ARG_DOCUMENT, idoc);
			viewer = (IDocumentProcessor) FunctorTools.launch("DocumentViewerFactory", viewerArgs);
			viewer.addNotificationListener(StoppedEvent.ID, new INotificationListener() {
				@Override
				public void handleEvent(Event event) {
					onViewerClosed();
				}
			});
			setStateText("viewing " + idoc.getLongName());
		} catch (FunctorCancelledException e) {
			return;
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (idoc != null) {
				// release reference counter
				// document is now held by the viewer!
				idoc.release(null);
			}
		}
	}

	protected void onOpen() {
		if (viewer != null) {
			viewer.stop();
		}
		IDocument idoc = null;
		try {
			idoc = load();
			if (idoc == null) {
				return;
			}
			IArgs args = new Args();
			args.put(IDocumentProcessor.ARG_DOCUMENT, idoc);
			viewer = (IDocumentProcessor) FunctorTools.launch("DocumentViewerFactory", args);
			viewer.addNotificationListener(StoppedEvent.ID, new INotificationListener() {
				@Override
				public void handleEvent(Event event) {
					onViewerClosed();
				}
			});
			setStateText("viewing " + idoc.getLongName());
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (idoc != null) {
				// release reference counter
				// document is now held by the viewer!
				idoc.release(null);
			}
		}
	}

	protected void onViewerClosed() {
		viewer = null;
		setStateText("viewer closed");
	}

	protected void setStateText(String text) {
		stateLabel.setText(text);
	}

	protected void shutdownContainer() {
		SingletonContainerLauncher.get().release();
	}
}
