/*
 *   o_
 * in|tarsys GmbH (c)
 *
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.auth;

import java.io.ByteArrayInputStream;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import javax.naming.ldap.LdapName;
import javax.security.auth.Subject;

import com.fasterxml.jackson.jakarta.rs.json.JacksonJsonProvider;

import de.intarsys.tools.functor.ArgTools;
import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.jaxrs.JaxrsTools;
import de.intarsys.tools.string.StringTools;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.client.Invocation;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status.Family;

public class SmartcardContext extends CommonAuthContext {

	private static final String PARAM_CALLBACK = "callback";

	private static final String PARAM_STATE = "state";

	private static final String PARAM_LABEL = "label";

	private static final String PARAM_DESCRIPTION = "description";

	private Client client;

	private String meetingId;

	public SmartcardContext(CommonAuthModule authModule, Subject subject, IArgs args) {
		super(authModule, subject, args);
		client = ClientBuilder.newBuilder() //
				.build() //
				.register(JacksonJsonProvider.class);
	}

	@Override
	public String authCallback(HttpServletRequest servletRequest) throws AuthenticationException {
		// collect the result (indirectly disposing the meeting)
		String uriMeeting = "http://localhost:8080/cloudsuite-gears/core/api/v1/bridglet/{meeting}/result";
		WebTarget targetMeeting = client
				.target(uriMeeting)
				.resolveTemplate("meeting", meetingId);
		Invocation.Builder builder = targetMeeting.request();
		Response response = builder.get();
		if (response.getStatusInfo().getFamily() != Family.SUCCESSFUL) {
			throw new AuthenticationFailed("authentication failed");
		}
		Map<String, String> result = response.readEntity(Map.class);
		String strCertificate = result.get("certificate");
		if (StringTools.isEmpty(strCertificate)) {
			throw new IllegalArgumentException("certificate required");
		}
		String strResponse = result.get("response");
		if (StringTools.isEmpty(strResponse)) {
			throw new IllegalArgumentException("response required");
		}
		byte[] byteCertificate = Base64.getDecoder().decode(strCertificate);
		try {
			CertificateFactory factory = CertificateFactory.getInstance("X509");
			X509Certificate certificate = (X509Certificate) factory.generateCertificate(new ByteArrayInputStream(
					byteCertificate));
			String name = certificate.getSubjectX500Principal().getName();
			LdapName ldapDN = new LdapName(name);
			String cn = ldapDN.getRdns().stream().filter((rdn) -> rdn.getType().equalsIgnoreCase("cn")).map(rdn -> rdn
					.getValue()
					.toString()).findFirst().orElse("unknown");
			SmartcardPrincipal principal = new SmartcardPrincipal(cn);
			getSubject().getPrincipals().add(principal);
			return principal.getToken();
		} catch (Exception e) {
			throw new AuthenticationFailed(e);
		}
	}

	@Override
	protected Response basicAuthenticate(HttpServletRequest servletRequest) throws AuthenticationException {
		/*
		 * set up a bridge authenticator meeting
		 */
		String uriMeeting = "http://localhost:8080/cloudsuite-gears/core/api/v1/bridglet/authenticator/create";
		WebTarget targetMeeting = client.target(uriMeeting);
		Invocation.Builder builder = targetMeeting.request();
		Object request = new HashMap<>();
		Response response = builder.post(Entity.entity(request, MediaType.APPLICATION_JSON_TYPE));
		if (response.getStatusInfo().getFamily() != Family.SUCCESSFUL) {
			throw new AuthenticationFailed("authentication failed");
		}
		Map<String, String> result = response.readEntity(Map.class);
		meetingId = result.get("id");

		/*
		 * redirect to bridge web page and provide authentication callback
		 * adress to resume
		 */
		String uriCallback = JaxrsTools.getUriBuilderContext(servletRequest).path("/api/v1/authentication/callback")
				.toString();
		String uriBridge = "http://localhost:8080/cloudsuite-gears/core/bridge/authenticate";
		WebTarget targetRedirect = client
				.target(uriBridge)
				.queryParam(PARAM_STATE, meetingId)
				.queryParam(PARAM_CALLBACK, uriCallback)
				.queryParam(PARAM_LABEL, ArgTools.getString(getArgs(), "label", "Authenticate with your smartcard"))
				.queryParam(PARAM_DESCRIPTION, ArgTools.getString(getArgs(), "description",
						"Using the local bridge agent you can authenticate with your local smartcard."));

		return Response.temporaryRedirect(targetRedirect.getUri()).build();
	}

	@Override
	protected void basicDeauthenticate() {
		getSubject().getPrincipals().removeIf((principal) -> principal instanceof SmartcardPrincipal);
	}

	@Override
	public SmartcardModule getAuthModule() {
		return (SmartcardModule) super.getAuthModule();
	}

}
