/*
 *   o_
 * in|tarsys GmbH (c)
 *
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.jaxrs;

import java.io.IOException;
import java.util.Locale;

import javax.security.auth.Subject;

import org.springframework.beans.factory.annotation.Autowired;

import de.intarsys.cloudsuite.gears.demo.auth.AuthenticationException;
import de.intarsys.cloudsuite.gears.demo.auth.AuthenticationFailed;
import de.intarsys.cloudsuite.gears.demo.auth.DemoPrincipal;
import de.intarsys.cloudsuite.gears.demo.model.AuthenticationBackend;
import de.intarsys.cloudsuite.gears.demo.model.DemoUser;
import de.intarsys.cloudsuite.gears.demo.model.DemoUserStore;
import de.intarsys.tools.string.StringTools;
import jakarta.annotation.Priority;
import jakarta.ws.rs.Priorities;
import jakarta.ws.rs.container.ContainerRequestContext;
import jakarta.ws.rs.container.ContainerRequestFilter;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.ext.Provider;

@Authenticated
@Provider
@Priority(Priorities.AUTHENTICATION)
public class AuthenticationFilter implements ContainerRequestFilter {

	public static final String BEARER = "Bearer";

	@Autowired
	private DemoUserStore userStore;

	@Autowired
	private AuthenticationBackend authenticationBackend;

	protected void authenticate(ContainerRequestContext requestContext) throws AuthenticationFailed {
		String authorization = requestContext.getHeaderString(HttpHeaders.AUTHORIZATION);
		String token = getBearerToken(authorization);
		if (token == null) {
			throw new AuthenticationFailed("authentication failed");
		}
		DemoPrincipal principal = getPrincipal();
		if (principal == null) {
			throw new AuthenticationFailed("authentication failed");
		}
		if (!token.equals(principal.getToken())) {
			throw new AuthenticationFailed("authentication failed");
		}
		DemoUser user = userStore.getUser(principal.getName());
		if (user == null) {
			throw new AuthenticationFailed("authentication failed");
		}
		requestContext.setProperty("user", user);
		requestContext.setProperty("principal", principal);
		DemoSecurityContext securityContext = new DemoSecurityContext(principal, BEARER,
				requestContext.getSecurityContext().isSecure());
		requestContext.setSecurityContext(securityContext);
	}

	@Override
	public void filter(ContainerRequestContext requestContext) throws IOException {
		try {
			authenticate(requestContext);
		} catch (AuthenticationException e) {
			replyRequestAuthentication(requestContext, BEARER);
		} catch (Exception e) {
			replyDeny(requestContext);
		}
	}

	public AuthenticationBackend getAuthenticationBackend() {
		return authenticationBackend;
	}

	protected String getBearerToken(String authorization) {
		if (StringTools.isEmpty(authorization)) {
			return null;
		}
		if (!authorization.toLowerCase(Locale.ROOT).startsWith(BEARER.toLowerCase(Locale.ROOT) + " ")) {
			return null;
		}
		return authorization.substring(BEARER.length()).trim();
	}

	protected DemoPrincipal getPrincipal() {
		if (getAuthenticationBackend().getAuthContext() == null
				|| getAuthenticationBackend().getAuthContext().getSubject() == null) {
			return null;
		}
		Subject subject = getAuthenticationBackend().getAuthContext().getSubject();
		return subject.getPrincipals().stream().filter(DemoPrincipal.class::isInstance).findFirst()
				.map(DemoPrincipal.class::cast).orElse(null);
	}

	protected void replyDeny(ContainerRequestContext requestContext) {
		Response response = Response //
				.status(Response.Status.FORBIDDEN) //
				.build();
		requestContext.abortWith(response);
	}

	protected void replyRequestAuthentication(ContainerRequestContext requestContext, String authenticationMethod) {
		Response response = Response //
				.status(Response.Status.UNAUTHORIZED)//
				.header(HttpHeaders.WWW_AUTHENTICATE, authenticationMethod) //
				.build();
		requestContext.abortWith(response);
	}
}
