/*
 *   o_
 * in|tarsys GmbH (c)
 *
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.model;

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.security.KeyStore;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.net.ssl.HostnameVerifier;

import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.client.RequestEntityProcessing;
import org.glassfish.jersey.logging.LoggingFeature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.annotation.ApplicationScope;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.jakarta.rs.json.JacksonXmlBindJsonProvider;

import de.intarsys.cloudsuite.gears.core.service.common.api.DtoArtifact;
import de.intarsys.cloudsuite.gears.core.service.common.api.DtoConfiguration;
import de.intarsys.cloudsuite.gears.core.service.common.api.DtoConfigurations;
import de.intarsys.cloudsuite.gears.core.service.common.api.DtoResultDocument;
import de.intarsys.cloudsuite.gears.core.service.common.api.FlowOptions;
import de.intarsys.cloudsuite.gears.core.service.common.api.IOptionSupport;
import de.intarsys.cloudsuite.gears.core.service.common.api.TransportDocument;
import de.intarsys.cloudsuite.gears.core.service.explorer.api.RequestExplorerCreate;
import de.intarsys.cloudsuite.gears.core.service.explorer.api.ResponseExplorerCreate;
import de.intarsys.cloudsuite.gears.core.service.explorer.api.ResultExplorer;
import de.intarsys.cloudsuite.gears.core.service.signer.api.RequestSignerCreate;
import de.intarsys.cloudsuite.gears.core.service.signer.api.ResponseSignerCreate;
import de.intarsys.cloudsuite.gears.core.service.signer.api.ResultSigner;
import de.intarsys.cloudsuite.gears.core.service.timestamper.api.RequestTimestamperCreate;
import de.intarsys.cloudsuite.gears.core.service.timestamper.api.ResponseTimestamperCreate;
import de.intarsys.cloudsuite.gears.core.service.timestamper.api.ResultTimestamper;
import de.intarsys.cloudsuite.gears.core.service.viewer.api.RequestViewerCreate;
import de.intarsys.cloudsuite.gears.core.service.viewer.api.ResponseViewerCreate;
import de.intarsys.cloudsuite.gears.core.service.viewer.api.ResultViewer;
import de.intarsys.cloudsuite.gears.demo.auth.DemoPrincipal;
import de.intarsys.cloudsuite.gears.demo.auth.KerberosPrincipal;
import de.intarsys.cloudsuite.gears.demo.auth.OpenIdConnectPrincipal;
import de.intarsys.cloudsuite.gears.demo.auth.UserPasswordPrincipal;
import de.intarsys.cloudsuite.gears.demo.service.DtoDocRef;
import de.intarsys.cloudsuite.gears.demo.service.ResponseDemoConversation;
import de.intarsys.conversation.service.client.api.ConversationalRequest;
import de.intarsys.conversation.service.client.api.ConversationalResponse;
import de.intarsys.conversation.service.client.api.DtoConversationSnapshot;
import de.intarsys.conversation.service.client.api.DtoErrorDetail;
import de.intarsys.conversation.service.client.api.DtoErrorStage;
import de.intarsys.conversation.service.client.api.DtoReplyStage;
import de.intarsys.conversation.service.client.api.DtoResultStage;
import de.intarsys.conversation.service.client.api.RequestAcknowledge;
import de.intarsys.conversation.service.client.api.RequestCancel;
import de.intarsys.conversation.service.client.api.ResponseAcknowledge;
import de.intarsys.conversation.service.client.api.ResponseCancel;
import de.intarsys.tools.conversation.ConversationExpired;
import de.intarsys.tools.exception.EncodedException;
import de.intarsys.tools.exception.ExceptionTools;
import de.intarsys.tools.file.PathTools;
import de.intarsys.tools.function.Throwing;
import de.intarsys.tools.function.Throwing.Consumer;
import de.intarsys.tools.functor.ArgTools;
import de.intarsys.tools.functor.Args;
import de.intarsys.tools.functor.IArgs;
import de.intarsys.tools.functor.IArgs.IBinding;
import de.intarsys.tools.jackson.DtoTypedObject;
import de.intarsys.tools.jaxrs.JaxrsTools;
import de.intarsys.tools.jaxrs.exception.ResponseError;
import de.intarsys.tools.json.Json;
import de.intarsys.tools.locator.FileLocator;
import de.intarsys.tools.locator.ILocator;
import de.intarsys.tools.locator.LocatorTools;
import de.intarsys.tools.stream.StreamTools;
import de.intarsys.tools.string.StringTools;
import jakarta.annotation.PostConstruct;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.client.Invocation;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status.Family;
import jakarta.ws.rs.core.SecurityContext;
import jakarta.ws.rs.core.StreamingOutput;

@Component
@ApplicationScope
public class DemoBackend {

	class ConversationState {
		private final DtoConversationSnapshot snapshot;
		private final Throwing.Specific.Consumer handler;

		public <T> ConversationState(DtoConversationSnapshot snapshot,
				Throwing.Specific.Consumer<T, Exception> handler) {
			super();
			this.snapshot = snapshot;
			this.handler = handler;
		}

		public <T> Throwing.Specific.Consumer<T, Exception> getHandler() {
			return handler;
		}

		public DtoConversationSnapshot getSnapshot() {
			return snapshot;
		}
	}

	private static final String DEBUG_URL_CLIENT = "debug.urlClient";

	private static final String HEADER_VALUE_CONTENT_DISPOSITION = "attachment; filename=\"%s\"";

	private static final String HEADER_CONTENT_DISPOSITION = "Content-Disposition";

	private static final String HEADER_CONTENT_TYPE = "Content-type";

	private static final String HEADER_VALUE_APPLICATION_OCTET_STREAM = "application/octet-stream";

	public static final String VAR_SIGNER_CREATE_ARGS = "signerCreate.args";

	public static final String VAR_SIGNER_CREATE_CONFIGURATION = "signerCreate.configuration";

	public static final String VAR_TIMESTAMPER_CREATE_CONFIGURATION = "timestamperCreate.configuration";

	public static final String SETTING_GEARS_CORE_SIGNER_CREATE_CONFIGURATION = "gears.core.signerCreate.configuration";

	public static final String SETTING_GEARS_CORE_TIMESTAMPER_CREATE_CONFIGURATION = "gears.core.timestamperCreate.configuration";

	public static final String SETTING_GEARS_CORE_VIEWER_CREATE_CONFIGURATION = "gears.core.viewerCreate.configuration";

	public static final String SETTING_GEARS_CORE_EXPLORER_CREATE_CONFIGURATION = "gears.core.explorerCreate.configuration";

	public static final String SETTING_GEARS_CORE_SIGNER_CREATE_ARGS = "gears.core.signerCreate.args";

	public static final String SETTING_GEARS_CORE_VIEWER_CREATE_ARGS = "gears.core.viewerCreate.args";

	public static final String SETTING_GEARS_CORE_OPTIONS = "gears.core.options";

	public static final String SETTING_GEARS_CORE_VARIABLES = "gears.core.variables";

	public static final String SETTING_GEARS_CORE_URL_SERVER = "gears.core.urlServer";

	/*
	 * one of none | basic
	 */
	public static final String SETTING_GEARS_CORE_AUTH_TYPE = "gears.core.auth.type";

	public static final String SETTING_GEARS_CORE_AUTH_BASIC_USER = "gears.core.auth.basic.user";

	public static final String SETTING_GEARS_CORE_AUTH_BASIC_PASSWORD = "gears.core.auth.basic.password";

	public static final String CSURL_PATH_PREFIX = "/api/v1/flow";

	public static final URI CSURL_EXPLORER_CREATE = URI.create("/explorer/create");

	public static final URI CSURL_CONVERSATION_ACKNOWLEDGE = URI.create("/conversation/acknowledge");

	public static final URI CSURL_CONVERSATION_CANCEL = URI.create("/conversation/cancel");

	public static final URI CSURL_VIEWER_CREATE = URI.create("/viewer/create");

	private static final URI CSURL_SIGNER_CREATE = URI.create("/signer/create");

	private static final URI CSURL_TIMESTAMPER_CREATE = URI.create("/timestamper/create");

	private Client client;

	@Value("${gears.core.urlServer:}")
	private String defaultGearsCoreUrlServer;

	@Value("${gears.core.ssl.keyStore.fileName:}")
	private String defaultGearsCoreSslKeyStoreFileName;

	@Value("${gears.core.ssl.keyStore.password:}")
	private String defaultGearsCoreSslKeyStorePassword;

	@Value("${gears.core.ssl.key.password:}")
	private String defaultGearsCoreSslKeyPassword;

	@Value("${gears.core.ssl.trustStore.fileName:}")
	private String defaultGearsCoreSslTrustStoreFileName;

	@Value("${gears.core.ssl.trustStore.password:}")
	private String defaultGearsCoreSslTrustStorePassword;

	@Value("${gears.core.signerCreate.configuration:}")
	private String defaultSignerCreateConfiguration;

	@Value("${gears.core.timestamperCreate.configuration:}")
	private String defaultTimestamperCreateConfiguration;

	@Value("${gears.core.viewerCreate.configuration:demoSignature}")
	private String defaultViewerCreateConfiguration;

	@Value("${gears.core.explorerCreate.configuration:demoSignature}")
	private String defaultExplorerCreateConfiguration;

	@Value("${gears.core.signerCreate.args:}")
	private String defaultSignerCreateArgs;

	@Value("${gears.core.viewerCreate.args:}")
	private String defaultViewerCreateArgs;

	@Value("${gears.core.options:}")
	private String defaultOptions;

	@Value("${gears.core.variables:}")
	private String defaultVariables;

	@Autowired(required = false)
	private HostnameVerifier gearsCoreSslHostnameVerifier;

	private Map<String, ConversationState> conversationStates = new HashMap<>();

	private KeyStore gearsCoreSslKeyStore;

	private KeyStore gearsCoreSslTrustStore;

	@Autowired
	private HttpServletRequest servletRequest;

	private SecurityContext securityContext;

	private final ObjectMapper mapper = new ObjectMapper();

	protected <T> T callBasic(URI endpoint, Map<String, String[]> parameters, Object request, Class<T> responseClass,
			CallModifier... modifiers) throws Exception {
		WebTarget target = createTarget(endpoint, parameters);
		Invocation.Builder builder = target.request() //
				.accept(MediaType.APPLICATION_JSON);
		try {
			for (CallModifier modifier : modifiers) {
				if (modifier != null) {
					modifier.accept(builder, target.getUri());
				}
			}
			Response response = builder.post(Entity.entity(request, MediaType.APPLICATION_JSON));
			if (response.getStatusInfo().getFamily().equals(Family.SUCCESSFUL)) {
				return response.readEntity(responseClass);
			} else {
				MediaType mediaType = response.getMediaType();
				if (response.getStatusInfo().equals(Response.Status.UNAUTHORIZED)) {
					throw new IOException("error " + response.getStatus() + " calling service '" + endpoint
							+ "', gears authentication failed");
				} else if (MediaType.APPLICATION_JSON_TYPE.equals(mediaType)) {
					ResponseError error = response.readEntity(ResponseError.class);
					throw new EncodedException(response.getStatus(), error.getError().getCode(),
							error.getError().getMessage());
				} else {
					throw new IOException("error " + response.getStatus() + " calling service '" + endpoint + "', "
							+ response.getStatusInfo().getReasonPhrase());
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw e;
		}
	}

	protected <T> T callWithOption(URI endpoint, IOptionSupport request, Class<T> responseClass,
			CallModifier... modifiers)
			throws Exception {
		FlowOptions.setRestrictedIdentification(request, Base64.getEncoder().encodeToString(getUser().getName()
				.getBytes()));
		FlowOptions.setPrincipal(request, toPrincipalOption(getPrincipal()));
		return callBasic(endpoint, null, request, responseClass, modifiers);
	}

	public ResponseDemoConversation conversationAcknowledge(RequestAcknowledge request,
			HttpServletRequest httpServletRequest) throws Exception {
		URI endpoint = lookupEndpoint(request, "acknowledge", CSURL_CONVERSATION_ACKNOWLEDGE);
		Map<String, String[]> params = httpServletRequest.getParameterMap();
		ResponseAcknowledge response = callBasic(endpoint, params, request, ResponseAcknowledge.class);
		return handleResponse(response, null);
	}

	public ResponseCancel conversationCancel(RequestCancel request,
			HttpServletRequest httpServletRequest) throws Exception {
		URI endpoint = lookupEndpoint(request, "cancel", CSURL_CONVERSATION_CANCEL);
		Map<String, String[]> params = httpServletRequest.getParameterMap();
		callBasic(endpoint, params, request, ResponseCancel.class);
		return new ResponseCancel();
	}

	protected Client createClient() {
		ClientBuilder builder = ClientBuilder.newBuilder();
		if (getGearsCoreSslKeyStore() != null) {
			builder = builder.keyStore(getGearsCoreSslKeyStore(), getGearsCoreSslKeyPassword());
		}
		if (getGearsCoreSslTrustStore() != null) {
			builder = builder.trustStore(getGearsCoreSslTrustStore());
		}
		if (getGearsCoreSslHostnameVerifier() != null) {
			builder = builder.hostnameVerifier(getGearsCoreSslHostnameVerifier());
		}
		builder.register(new LoggingFeature(Logger.getLogger("jaxrs"), Level.FINER,
				LoggingFeature.Verbosity.PAYLOAD_ANY,
				1000));

		Client result = builder //
				.build() //
				.register(new JacksonXmlBindJsonProvider())
				.property(ClientProperties.REQUEST_ENTITY_PROCESSING, RequestEntityProcessing.CHUNKED);
		return result;
	}

	protected WebTarget createTarget(URI endpoint, Map<String, String[]> parameters) {
		WebTarget target;
		if (endpoint.isAbsolute()) {
			target = getClient() //
					.target(endpoint);
		} else {
			target = getClient() //
					.target(getGearsCoreUrlServer()) //
					.path(CSURL_PATH_PREFIX) //
					.path(endpoint.getPath());
		}
		if (parameters != null) {
			for (Map.Entry<String, String[]> entry : parameters.entrySet()) {
				target = target.queryParam(entry.getKey(), (Object[]) entry.getValue());
			}
		}
		return target;
	}

	public void deleteAllDocuments() throws IOException {
		getUser().deleteAllDocuments();
	}

	public void deleteDocument(String name) throws IOException {
		getUser().deleteDocument(name);
	}

	public void deleteDocuments(List<DtoDocRef> docRefs) throws IOException {
		List<String> documents = docRefs.stream().map(DtoDocRef::getName).toList();
		getUser().deleteDocuments(documents);
	}

	public Response downloadDocument(DtoDocRef ref) throws IOException {
		DemoDoc doc = getDocument(ref.getName());
		StreamingOutput stream = new StreamingOutput() {
			@Override
			public void write(OutputStream os) throws IOException, WebApplicationException {
				InputStream is = new FileInputStream(doc.getPath());
				try {
					StreamTools.copy(is, os);
				} finally {
					StreamTools.close(is);
				}
			}
		};
		String filename = PathTools.getName(doc.getName());
		return Response //
				.ok(stream) //
				.header(HEADER_CONTENT_TYPE, HEADER_VALUE_APPLICATION_OCTET_STREAM) //
				.header(HEADER_CONTENT_DISPOSITION, String.format(HEADER_VALUE_CONTENT_DISPOSITION, filename)) //
				.build();
	}

	public Response downloadDocuments(List<DtoDocRef> documents) throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		ZipOutputStream zos = new ZipOutputStream(baos);
		try {
			for (DtoDocRef doc : documents) {
				InputStream is = null;
				try {
					DemoDoc realDoc = getDocument(doc.getName());
					zos.putNextEntry(new ZipEntry(realDoc.getName()));
					is = realDoc.getLocator().getInputStream();
					StreamTools.copy(is, zos);
					zos.closeEntry();
				} finally {
					StreamTools.close(is);
				}
			}
		} finally {
			zos.close();
		}
		StreamingOutput stream = new StreamingOutput() {
			@Override
			public void write(OutputStream os) throws IOException, WebApplicationException {
				os.write(baos.toByteArray());
			}
		};
		String filename = PathTools.getName("folder.zip");
		return Response //
				.ok(stream) //
				.header(HEADER_CONTENT_TYPE, HEADER_VALUE_APPLICATION_OCTET_STREAM) //
				.header(HEADER_CONTENT_DISPOSITION, "attachment; filename=\"" + filename + "\"") //
				.build();
	}

	public void editDocument(DtoDocRef ref) throws Exception {
		DemoDoc document = getDocument(ref.getName());
		try {
			document.setProperties(ref.getProperties());
			getUser().save();
		} catch (Exception e) {
			document.setProperties(document.getProperties());
			throw e;
		}
	}

	protected Client getClient() {
		return client;
	}

	protected String getDebugGearsCoreUrlClient() {
		return resolvePreset().getSettings().get("debug.gears.core.urlClient");
	}

	public String getDefaultExplorerCreateConfiguration() {
		return defaultExplorerCreateConfiguration;
	}

	public String getDefaultGearsCoreSslKeyPassword() {
		return defaultGearsCoreSslKeyPassword;
	}

	public String getDefaultGearsCoreSslKeyStoreFileName() {
		return defaultGearsCoreSslKeyStoreFileName;
	}

	public String getDefaultGearsCoreSslKeyStorePassword() {
		return defaultGearsCoreSslKeyStorePassword;
	}

	public String getDefaultGearsCoreSslTrustStoreFileName() {
		return defaultGearsCoreSslTrustStoreFileName;
	}

	public String getDefaultGearsCoreSslTrustStorePassword() {
		return defaultGearsCoreSslTrustStorePassword;
	}

	public String getDefaultGearsCoreUrlServer() {
		return defaultGearsCoreUrlServer;
	}

	public String getDefaultOptions() {
		return defaultOptions;
	}

	public String getDefaultSignerCreateArgs() {
		return defaultSignerCreateArgs;
	}

	public String getDefaultSignerCreateConfiguration() {
		return defaultSignerCreateConfiguration;
	}

	public String getDefaultTimestamperCreateConfiguration() {
		return defaultTimestamperCreateConfiguration;
	}

	public String getDefaultVariables() {
		return defaultVariables;
	}

	public String getDefaultViewerCreateArgs() {
		return defaultViewerCreateArgs;
	}

	public String getDefaultViewerCreateConfiguration() {
		return defaultViewerCreateConfiguration;
	}

	public DemoDoc getDocument(String name) throws IOException {
		return getUser().getDocument(name);
	}

	public Collection<DemoDoc> getDocuments() {
		return getUser().getDocuments().values();
	}

	protected String getExplorerCreateConfigurationDefinition() {
		String configuration = resolvePreset().getSettings().get(SETTING_GEARS_CORE_EXPLORER_CREATE_CONFIGURATION);
		return StringTools.isEmpty(configuration)
				? getDefaultExplorerCreateConfiguration()
				: configuration.trim();
	}

	protected CallModifier getGearsCoreAuthModifier() {
		String type = resolvePreset().getSettings().get(SETTING_GEARS_CORE_AUTH_TYPE);
		if (StringTools.isEmpty(type)) {
			return null;
		} else if ("none".equals(type)) {
			return null;
		} else if ("forward".equals(type)) {
			Principal principal = getSecurityContext().getUserPrincipal();
			if (principal instanceof UserPasswordPrincipal) {
				String user = principal.getName();
				String password = ((UserPasswordPrincipal) principal).getPassword();
				if (StringTools.isEmpty(user) || StringTools.isEmpty(password)) {
					return null;
				}
				return new BasicAuthModifier(user, password);
			} else if (principal instanceof KerberosPrincipal) {
				return new KerberosModifier(((KerberosPrincipal) principal).getDelegCred());
			} else if (principal instanceof OpenIdConnectPrincipal) {
				return new OAuth2Modifier(((OpenIdConnectPrincipal) principal).getIdToken());
			} else {
				throw new IllegalArgumentException("must use user/password authentiation for forwarding");
			}
		} else if ("basic".equals(type)) {
			String user = resolvePreset().getSettings().get(SETTING_GEARS_CORE_AUTH_BASIC_USER);
			String password = resolvePreset().getSettings().get(SETTING_GEARS_CORE_AUTH_BASIC_PASSWORD);
			if (StringTools.isEmpty(user) || StringTools.isEmpty(password)) {
				return null;
			}
			return new BasicAuthModifier(user, password);
		}
		throw new IllegalArgumentException("unsupported auth type");
	}

	protected HostnameVerifier getGearsCoreSslHostnameVerifier() {
		return gearsCoreSslHostnameVerifier;
	}

	protected String getGearsCoreSslKeyPassword() {
		return getDefaultGearsCoreSslKeyPassword();
	}

	protected KeyStore getGearsCoreSslKeyStore() {
		if (gearsCoreSslKeyStore == null) {
			gearsCoreSslKeyStore = loadKeyStore(getGearsCoreSslKeyStoreFileName(), getGearsCoreSslKeyStorePassword());
		}
		return gearsCoreSslKeyStore;
	}

	protected String getGearsCoreSslKeyStoreFileName() {
		return getDefaultGearsCoreSslKeyStoreFileName();
	}

	protected String getGearsCoreSslKeyStorePassword() {
		return getDefaultGearsCoreSslKeyStorePassword();
	}

	protected KeyStore getGearsCoreSslTrustStore() {
		if (gearsCoreSslTrustStore == null) {
			gearsCoreSslTrustStore = loadKeyStore(getGearsCoreSslTrustStoreFileName(),
					getGearsCoreSslTrustStorePassword());
		}
		return gearsCoreSslTrustStore;
	}

	protected String getGearsCoreSslTrustStoreFileName() {
		return getDefaultGearsCoreSslTrustStoreFileName();
	}

	protected String getGearsCoreSslTrustStorePassword() {
		return getDefaultGearsCoreSslTrustStorePassword();
	}

	protected String getGearsCoreUrlServer() {
		String result = resolvePreset().getSettings().get(SETTING_GEARS_CORE_URL_SERVER);
		if (StringTools.isEmpty(result)) {
			result = getDefaultGearsCoreUrlServer();
			if (StringTools.isEmpty(result)) {
				result = JaxrsTools.getUriBuilderRoot(servletRequest).path("/cloudsuite-gears/core").toString();
			}
		}
		return result;
	}

	public ObjectMapper getMapper() {
		return mapper;
	}

	protected String getOptionsDefinition() {
		String result = resolvePreset().getSettings().get(SETTING_GEARS_CORE_OPTIONS);
		if (StringTools.isEmpty(result)) {
			result = getDefaultOptions();
		}
		return result == null ? "" : result;
	}

	protected IArgs getOptionsExpanded() {
		return toArgs(getOptionsDefinition());
	}

	public List<DemoPreset> getPresets() {
		return getUser().getPresets();
	}

	protected Principal getPrincipal() {
		return (Principal) servletRequest.getAttribute("principal");
	}

	public SecurityContext getSecurityContext() {
		return securityContext;
	}

	public String getSelectedPreset() {
		return getUser().getSelectedPreset();
	}

	protected HttpServletRequest getServletRequest() {
		return servletRequest;
	}

	protected String getSignerCreateArgsDefinition() {
		String result = resolvePreset().getSettings().get(SETTING_GEARS_CORE_SIGNER_CREATE_ARGS);
		if (StringTools.isEmpty(result)) {
			result = getDefaultSignerCreateArgs();
		}
		return result == null ? "" : result;
	}

	protected IArgs getSignerCreateArgsExpanded() {
		return toArgs(getSignerCreateArgsDefinition());
	}

	protected String getSignerCreateConfigurationDefinition() {
		String configuration = resolvePreset().getSettings().get(SETTING_GEARS_CORE_SIGNER_CREATE_CONFIGURATION);
		return StringTools.isEmpty(configuration)
				? getDefaultSignerCreateConfiguration()
				: configuration.trim();
	}

	protected String getTimestamperCreateConfigurationDefinition() {
		String result = resolvePreset().getSettings().get(SETTING_GEARS_CORE_TIMESTAMPER_CREATE_CONFIGURATION);
		if (StringTools.isEmpty(result)) {
			result = getDefaultTimestamperCreateConfiguration();
		}
		return result;
	}

	protected Object getSignerCreateConfigurationExpanded() {
		String result = getSignerCreateConfigurationDefinition();
		if (Json.isJson(result)) {
			try {
				return getMapper().readValue(result, Object.class);
			} catch (Exception e) {
				throw new IllegalArgumentException("signer configuration invalid (" + ExceptionTools.getMessage(e)
						+ ")", e);
			}
		} else {
			return result;
		}
	}

	protected Object getTimestamperCreateConfigurationExpanded() {
		String result = getTimestamperCreateConfigurationDefinition();
		if (Json.isJson(result)) {
			try {
				return getMapper().readValue(result, Object.class);
			} catch (Exception e) {
				throw new IllegalArgumentException("timestamper configuration invalid (" + ExceptionTools.getMessage(e)
						+ ")", e);
			}
		} else {
			return result;
		}
	}

	protected DemoUser getUser() {
		return (DemoUser) servletRequest.getAttribute("user");
	}

	protected String getVariablesDefinition() {
		String result = resolvePreset().getSettings().get(SETTING_GEARS_CORE_VARIABLES);
		if (StringTools.isEmpty(result)) {
			result = getDefaultVariables();
		}
		return result == null ? "" : result;
	}

	protected IArgs getVariablesExpanded() {
		return toArgs(getVariablesDefinition());
	}

	protected String getViewerCreateArgsDefinition() {
		String result = resolvePreset().getSettings().get(SETTING_GEARS_CORE_VIEWER_CREATE_ARGS);
		if (StringTools.isEmpty(result)) {
			result = getDefaultViewerCreateArgs();
		}
		return result == null ? "" : result;
	}

	protected IArgs getViewerCreateArgsExpanded() {
		IArgs args = toArgs(getViewerCreateArgsDefinition());
		/*
		 * This is a well known argument for the configuration editor, do not confound with "configuration"
		 */
		Object configurations = ArgTools.getObject(args, "configurations", null);
		if (configurations != null) {
			if ("*".equals(configurations)) {
				args.put("configurations", getUser().getConfigurations());
			} else if (configurations instanceof String) {
				args.put("configurations", resolveConfig(configurations));
			} else if (configurations instanceof IArgs) {
				for (IBinding binding : (IArgs) configurations) {
					binding.setValue(resolveConfig(binding.getValue()));
				}
			} else {
				// ?
			}
		}
		return args;
	}

	protected String getViewerCreateConfigurationDefinition() {
		String configuration = resolvePreset().getSettings().get(SETTING_GEARS_CORE_VIEWER_CREATE_CONFIGURATION);
		return StringTools.isEmpty(configuration)
				? getDefaultViewerCreateConfiguration()
				: configuration.trim();
	}

	protected Object getViewerCreateConfigurationExpanded() {
		String result = getViewerCreateConfigurationDefinition();
		if (Json.isJson(result)) {
			Object jsonObject;
			try {
				jsonObject = getMapper().readValue(result, Object.class);
			} catch (Exception e) {
				throw new IllegalArgumentException("viewer configuration invalid (" + ExceptionTools.getMessage(e)
						+ ")", e);
			}
			if (jsonObject instanceof List<?> list) {
				return list.stream()
						.map(this::resolveConfig)
						.toList();
			}
			return jsonObject;
		}

		return resolveConfig(result);
	}

	protected ResponseDemoConversation handleResponse(ConversationalResponse response, Consumer consumer) {
		DtoConversationSnapshot snapshot = response.getSnapshot();
		if (consumer != null) {
			conversationStates.put(snapshot.getConversationHandle(), new ConversationState(snapshot, consumer));
		}
		ResponseDemoConversation responseDemo = new ResponseDemoConversation();
		DtoConversationSnapshot mySnapshot = snapshot;
		DtoReplyStage replyStage = snapshot.getReplyStage();
		if (replyStage.isFinal()) {
			ConversationState state = conversationStates.remove(snapshot.getConversationHandle());
			if (state != null && state.getHandler() != null && replyStage instanceof DtoResultStage) {
				mySnapshot = new DtoConversationSnapshot();
				mySnapshot.setConversationHandle(snapshot.getConversationHandle());
				try {
					state.getHandler().accept(((DtoResultStage) replyStage).getResult());
					// do not stream result objects, not needed on ajax client
					mySnapshot.setReplyStage(new DtoResultStage(null));
				} catch (Exception e) {
					mySnapshot.setReplyStage(new DtoErrorStage(new DtoErrorDetail("HandleResponse", e.getMessage())));
				}
			}
		}
		responseDemo.setSnapshot(mySnapshot);
		return responseDemo;
	}

	public DemoDoc importDocument(String filename, ILocator locator) throws IOException {
		return getUser().importDocument(filename, locator);
	}

	@PostConstruct
	public void initialize() {
		client = createClient();
	}

	protected KeyStore loadKeyStore(String name, String password) {
		if (StringTools.isEmpty(name)) {
			return null;
		}

		try (InputStream is = new FileInputStream(name)) {
			KeyStore ks = KeyStore.getInstance("JKS");
			ks.load(is, password.toCharArray());
			return ks;
		} catch (Exception e) {
			throw new IllegalArgumentException("cannot load keystore " + name, e);
		}
	}

	protected URI lookupEndpoint(ConversationalRequest request, String action, URI defaultValue) {
		ConversationState state = conversationStates.get(request.getConversationHandle());
		if (state == null) {
			throw new ConversationExpired("unknown conversation '" + request.getConversationHandle() + "'");
		}
		String endpoint = state.getSnapshot().getLink(action);
		if (endpoint == null) {
			return defaultValue;
		}
		return URI.create(endpoint);
	}

	protected void onResultArtifact(DtoArtifact artifact) throws IOException {
		Object value = artifact.getValue();
		if (value instanceof ResultSigner) {
			onResultSigner((ResultSigner) value);
		}
		if (value instanceof ResultTimestamper) {
			onResultTimestamper((ResultTimestamper) value);
		}
		if (value instanceof DtoResultDocument) {
			onResultDocument((DtoResultDocument) value);
		}
		if (value instanceof ResultViewer) {
			onResultViewer((ResultViewer) value);
		}
		if (value instanceof DtoConfigurations) {
			onResultConfigurations((DtoConfigurations) value);
		}
		if (value instanceof DtoTypedObject) {
			// local implementation of target class missing
		}
	}

	protected void onResultArtifacts(List<DtoArtifact> artifacts) throws IOException {
		if (artifacts == null) {
			return;
		}
		for (DtoArtifact artifact : artifacts) {
			onResultArtifact(artifact);
		}
	}

	protected void onResultConfigurations(DtoConfigurations value) throws IOException {
		setConfigurations(value);
	}

	protected void onResultDocument(DtoResultDocument result) throws IOException {
		if (result.getDocument() == null) {
			return;
		}
		String name = result.getDocument().getName();
		ILocator content = result.getDocument().getLocator();
		try {
			LocatorTools.copy(content, getDocument(name).getLocator());
		} finally {
			content.delete();
		}
		getUser().save();
	}

	protected void onResultExplorer(ResultExplorer result) throws IOException {
		onResultArtifacts(result.getArtifacts());
	}

	protected void onResultSigner(ResultSigner result) throws IOException {
		if (result.getSignatures() == null) {
			return;
		}
		for (TransportDocument transportItem : result.getSignatures()) {
			onResultSignerDocument(transportItem);
		}
		getUser().save();
	}

	protected void onResultTimestamper(ResultTimestamper result) throws IOException {
		if (result.getDocumentTimestamps() == null) {
			return;
		}
		for (TransportDocument transportItem : result.getDocumentTimestamps()) {
			onResultTimestamperDocument(transportItem);
		}
		getUser().save();
	}

	protected void onResultSignerDocument(TransportDocument transportDoc) throws IOException {
		String name = transportDoc.getName();
		String targetName = (String) transportDoc.getProperty("signature.targetName");
		ILocator content = transportDoc.getLocator();
		try {
			DemoDoc demoDoc = getDocument(targetName);
			demoDoc.setSigned(true);
			try {
				LocatorTools.copy(content, getDocument(name).getLocator());
			} catch (FileNotFoundException e) {
				/*
				 * detached signature
				 */
				getUser().importDocument(name, content);
			}
		} finally {
			content.delete();
		}
	}

	protected void onResultTimestamperDocument(TransportDocument transportDoc) throws IOException {
		String name = transportDoc.getName();
		String targetName = (String) transportDoc.getProperty("timestamp.targetName");
		ILocator content = transportDoc.getLocator();
		try {
			DemoDoc demoDoc = getDocument(targetName);
			demoDoc.setTimestamped(true);
			try {
				LocatorTools.copy(content, getDocument(name).getLocator());
			} catch (FileNotFoundException e) {
				getUser().importDocument(name, content);
			}
		} finally {
			content.delete();
		}
	}

	protected void onResultViewer(ResultViewer result) throws IOException {
		DemoDoc doc = getDocument(result.getDocumentName());
		doc.setViewed(true);
		getUser().save();
		onResultArtifacts(result.getArtifacts());
	}

	public ResponseDemoConversation openDocument(DtoDocRef ref, String redirectUri) throws Exception {
		RequestViewerCreate reqCreate = new RequestViewerCreate();
		reqCreate.setArgs(getViewerCreateArgsExpanded());
		reqCreate.setOptions(getOptionsExpanded());
		reqCreate.setConfiguration(getViewerCreateConfigurationExpanded());
		if (!StringTools.isEmpty(getDebugGearsCoreUrlClient())) {
			reqCreate.setOption(DEBUG_URL_CLIENT, getDebugGearsCoreUrlClient());
		}
		reqCreate.setOption("redirectUri", redirectUri);
		TransportDocument transportDocument = toTransportDocument(ref);
		reqCreate.setDocument(transportDocument);
		reqCreate.setVariables(getVariablesExpanded());
		ArgTools.putPathIfAbsent(reqCreate.getVariables(), VAR_SIGNER_CREATE_CONFIGURATION,
				getSignerCreateConfigurationExpanded());
		ArgTools.putPathIfAbsent(reqCreate.getVariables(), VAR_TIMESTAMPER_CREATE_CONFIGURATION,
				getTimestamperCreateConfigurationExpanded());
		ArgTools.putPathIfAbsent(reqCreate.getVariables(), VAR_SIGNER_CREATE_ARGS, getSignerCreateArgsExpanded());
		/*
		 * request viewer flow from cloud suite
		 */
		ResponseViewerCreate resCreate = callWithOption(CSURL_VIEWER_CREATE, reqCreate, ResponseViewerCreate.class,
				getGearsCoreAuthModifier());
		return handleResponse(resCreate, (result) -> onResultViewer((ResultViewer) result));
	}

	public ResponseDemoConversation openDocuments(List<DtoDocRef> documents, String redirectUri) throws Exception {
		RequestExplorerCreate reqCreate = new RequestExplorerCreate();
		reqCreate.setOptions(getOptionsExpanded());
		reqCreate.setConfiguration(getExplorerCreateConfigurationDefinition());
		if (!StringTools.isEmpty(getDebugGearsCoreUrlClient())) {
			reqCreate.setOption(DEBUG_URL_CLIENT, getDebugGearsCoreUrlClient());
		}
		reqCreate.setOption("redirectUri", redirectUri);
		List<TransportDocument> transportList = new ArrayList<>();
		for (Iterator iterator = documents.iterator(); iterator.hasNext();) {
			DtoDocRef clientDoc = (DtoDocRef) iterator.next();
			DemoDoc realDoc = getDocument(clientDoc.getName());
			IArgs args = ArgTools.toArgs(clientDoc.getProperties().get("properties"));
			TransportDocument transportDocument = TransportDocument.builder()
					.content(new FileLocator(realDoc.getPath()))
					.build();
			transportDocument.addProperties(args);
			transportList.add(transportDocument);
		}
		reqCreate.setDocuments(transportList);
		reqCreate.setVariables(getVariablesExpanded());
		ArgTools.putPath(reqCreate.getVariables(), VAR_SIGNER_CREATE_CONFIGURATION,
				getSignerCreateConfigurationExpanded());
		ArgTools.putPath(reqCreate.getVariables(), VAR_TIMESTAMPER_CREATE_CONFIGURATION,
				getTimestamperCreateConfigurationExpanded());
		ArgTools.putPath(reqCreate.getVariables(), VAR_SIGNER_CREATE_ARGS, getSignerCreateArgsExpanded());
		ResponseExplorerCreate resCreate = callWithOption(CSURL_EXPLORER_CREATE, reqCreate,
				ResponseExplorerCreate.class, getGearsCoreAuthModifier());
		return handleResponse(resCreate, (result) -> onResultExplorer((ResultExplorer) result));
	}

	/**
	 * Lets look if we can resolve {@code configObject} against a locally stored
	 * configuration instance.
	 *
	 * @param result
	 * @return
	 */
	protected Object resolveConfig(Object configObject) {
		if (configObject instanceof String && getUser().getConfigurations() != null) {
			String configId = (String) configObject;
			DtoConfiguration configValue = getUser().getConfigurations().getConfiguration(configId);
			if (configValue != null) {
				return configValue;
			}
		}
		return configObject;
	}

	protected DemoPreset resolvePreset() {
		return getUser().resolvePreset();
	}

	public void setConfigurations(DtoConfigurations configurations) throws IOException {
		getUser().setConfigurations(configurations);
		getUser().save();
	}

	public void setDefaultExplorerCreateConfiguration(String defaultExplorerConfiguration) {
		this.defaultExplorerCreateConfiguration = defaultExplorerConfiguration;
	}

	public void setDefaultGearsCoreSslKeyPassword(String defaultGearsCoreSslKeyPassword) {
		this.defaultGearsCoreSslKeyPassword = defaultGearsCoreSslKeyPassword;
	}

	public void setDefaultGearsCoreSslKeyStoreFileName(String value) {
		this.defaultGearsCoreSslKeyStoreFileName = value;
	}

	public void setDefaultGearsCoreSslKeyStorePassword(String value) {
		this.defaultGearsCoreSslKeyStorePassword = value;
	}

	public void setDefaultGearsCoreSslTrustStoreFileName(String value) {
		this.defaultGearsCoreSslTrustStoreFileName = value;
	}

	public void setDefaultGearsCoreSslTrustStorePassword(String value) {
		this.defaultGearsCoreSslTrustStorePassword = value;
	}

	public void setDefaultGearsCoreUrlServer(String defaultCloudsuiteUrl) {
		this.defaultGearsCoreUrlServer = defaultCloudsuiteUrl;
	}

	public void setDefaultOptions(String defaultSignerCreateOptions) {
		this.defaultOptions = defaultSignerCreateOptions;
	}

	public void setDefaultSignerCreateArgs(String defaultSignerArgs) {
		this.defaultSignerCreateArgs = defaultSignerArgs;
	}

	public void setDefaultSignerCreateConfiguration(String defaultSignerCreateConfiguration) {
		this.defaultSignerCreateConfiguration = defaultSignerCreateConfiguration;
	}

	public void setDefaultVariables(String defaultVariables) {
		this.defaultVariables = defaultVariables;
	}

	public void setDefaultViewerCreateArgs(String defaultViewerCreateArgs) {
		this.defaultViewerCreateArgs = defaultViewerCreateArgs;
	}

	public void setDefaultViewerCreateConfiguration(String defaultViewerConfiguration) {
		this.defaultViewerCreateConfiguration = defaultViewerConfiguration;
	}

	public void setPresets(List<DemoPreset> presets) throws Exception {
		List<DemoPreset> oldPresets = getPresets();
		try {
			getUser().setPresets(presets);
			getUser().save();
		} catch (Exception e) {
			getUser().setPresets(oldPresets);
			throw e;
		}
	}

	public void setSecurityContext(SecurityContext securityContext) {
		this.securityContext = securityContext;
	}

	public void setSelectedPreset(String name) throws Exception {
		getUser().setSelectedPreset(name);
		getUser().save();
	}

	public ResponseDemoConversation signDocument(DtoDocRef ref, String redirectUri) throws Exception {
		return signDocuments(List.of(ref), redirectUri);
	}

	public ResponseDemoConversation signDocuments(List<DtoDocRef> documents, String redirectUri) throws Exception {
		RequestSignerCreate reqCreate = new RequestSignerCreate();
		reqCreate.setConfiguration(getSignerCreateConfigurationExpanded());
		reqCreate.setOptions(getOptionsExpanded());
		reqCreate.setOption("redirectUri", redirectUri);
		reqCreate.setVariables(getVariablesExpanded());
		reqCreate.setArgs(getSignerCreateArgsExpanded());
		reqCreate.setDocuments(toTransportDocumentList(documents));

		/*
		 * request signer flow from cloud suite
		 */
		ResponseSignerCreate resCreate =
				callWithOption(CSURL_SIGNER_CREATE, reqCreate, ResponseSignerCreate.class, getGearsCoreAuthModifier());
		return handleResponse(resCreate, (result) -> onResultSigner((ResultSigner) result));
	}

	public ResponseDemoConversation timestampDocument(DtoDocRef ref, String redirectUri) throws Exception {
		RequestTimestamperCreate reqCreate = new RequestTimestamperCreate();
		reqCreate.setConfiguration(getTimestamperCreateConfigurationExpanded());
		reqCreate.setOptions(getOptionsExpanded());
		reqCreate.setOption("redirectUri", redirectUri);
		reqCreate.setVariables(getVariablesExpanded());
		reqCreate.setArgs(getSignerCreateArgsExpanded());
		reqCreate.getDocuments().add(toTransportDocument(ref));

		/*
		 * request timestamper flow from cloud suite
		 */
		ResponseTimestamperCreate resCreate = callWithOption(CSURL_TIMESTAMPER_CREATE, reqCreate,
				ResponseTimestamperCreate.class,
				getGearsCoreAuthModifier());
		return handleResponse(resCreate, (result) -> onResultTimestamper((ResultTimestamper) result));
	}

	public ResponseDemoConversation timestampDocuments(List<DtoDocRef> documents, String redirectUri) throws Exception {
		RequestTimestamperCreate reqCreate = new RequestTimestamperCreate();
		reqCreate.setConfiguration(getTimestamperCreateConfigurationExpanded());
		reqCreate.setOptions(getOptionsExpanded());
		reqCreate.setOption("redirectUri", redirectUri);
		reqCreate.setVariables(getVariablesExpanded());
		reqCreate.setArgs(getSignerCreateArgsExpanded());
		reqCreate.setDocuments(toTransportDocumentList(documents));

		/*
		 * request timestamper flow from cloud suite
		 */
		ResponseTimestamperCreate resCreate = callWithOption(CSURL_TIMESTAMPER_CREATE, reqCreate,
				ResponseTimestamperCreate.class,
				getGearsCoreAuthModifier());
		return handleResponse(resCreate, (result) -> onResultTimestamper((ResultTimestamper) result));
	}

	protected IArgs toArgs(String value) {
		return StringTools.isEmpty(value) ? Args.create() : ArgTools.toArgs(value);
	}

	protected IArgs toPrincipalOption(Principal principal) {
		if (principal == null) {
			return null;
		}
		IArgs argsPrincipal = Args.create();
		argsPrincipal.put("name", principal.getName());
		IArgs argsClaims = Args.create();
		argsPrincipal.put("claims", argsClaims);
		ArgTools.putAll(argsClaims, ArgTools.toArgs(resolvePreset().getSettings().get("demo.principal.claims")));
		if (principal instanceof DemoPrincipal) {
			ArgTools.putAll(argsClaims, ((DemoPrincipal) principal).getClaims());
		}
		return argsPrincipal;
	}

	protected TransportDocument toTransportDocument(DtoDocRef clientDoc) throws IOException {
		DemoDoc doc = getDocument(clientDoc.getName());
		TransportDocument transportDocument = new TransportDocument();
		transportDocument.setContent(new FileLocator(doc.getPath()));
		transportDocument.addProperties(ArgTools.toArgs(clientDoc.getProperties().get("properties")));
		return transportDocument;
	}

	protected List<TransportDocument> toTransportDocumentList(List<DtoDocRef> clientDocs) throws IOException {
		List<TransportDocument> result = new ArrayList<>();
		for (DtoDocRef clientDoc : clientDocs) {
			result.add(toTransportDocument(clientDoc));
		}
		return result;
	}

}
