/*
 *   o_
 * in|tarsys GmbH (c)
 *
 * all rights reserved
 *
 */
package de.intarsys.cloudsuite.gears.demo.model;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;

import de.intarsys.cloudsuite.gears.core.service.common.api.DtoConfigurations;
import de.intarsys.tools.file.FileTools;
import de.intarsys.tools.file.PathTools;
import de.intarsys.tools.locator.FileLocator;
import de.intarsys.tools.locator.ILocator;
import de.intarsys.tools.locator.LocatorTools;
import de.intarsys.tools.string.StringTools;

public class DemoUser {

	private static final String PATH_REPO = "repo";

	private static final String PATH_CONFIG = "config";

	public static final String FILENAME_USER = "user.json";

	private final DemoUserStore store;

	protected String name;

	@JsonInclude(Include.NON_EMPTY)
	protected List<DemoPreset> presets;

	protected String selectedPreset;

	/**
	 * @deprecated use {@link #presets}
	 */
	@Deprecated
	@JsonInclude(Include.NON_EMPTY)
	protected Map<String, String> settings;

	protected Map<String, DemoDoc> documents;

	protected DtoConfigurations configurations;

	public DemoUser() {
		this.store = null;
	}

	public DemoUser(DemoUserStore store, String name) {
		this.store = store;
		this.name = name;

		try {
			load();
			if (purgeObsoleteDocuments()) {
				save();
			}
		} catch (IOException exception) {
			exception.printStackTrace();
		}
	}

	public synchronized void deleteAllDocuments() throws IOException {
		boolean wasEmpty = documents.isEmpty();
		documents.clear();

		if (!wasEmpty | purgeObsoleteDocuments()) {
			save();
		}
	}

	public synchronized void deleteDocument(String name) throws IOException {
		if (documents.remove(name) != null | purgeObsoleteDocuments()) {
			save();
		}
	}

	public synchronized void deleteDocuments(Collection<String> names) throws IOException {
		boolean changed = documents.keySet().removeAll(names);
		changed |= purgeObsoleteDocuments();
		if (changed) {
			save();
		}
	}

	public DtoConfigurations getConfigurations() {
		return configurations;
	}

	public synchronized DemoDoc getDocument(String name) throws FileNotFoundException {
		DemoDoc document = documents.get(name);
		if (document == null) {
			throw new FileNotFoundException("document " + name + " not found");
		}

		return document;
	}

	public synchronized Map<String, DemoDoc> getDocuments() {
		return documents;
	}

	@JsonIgnore
	private File getHomeDir() {
		return new File(getStore().getRootDir(), FileTools.trimPath(getName()));
	}

	@JsonIgnore
	public String getName() {
		return name;
	}

	public List<DemoPreset> getPresets() {
		return presets;
	}

	private File getRepoDir() {
		return new File(getHomeDir(), PATH_REPO);
	}

	public String getSelectedPreset() {
		return selectedPreset;
	}

	public Map<String, String> getSettings() {
		return settings;
	}

	@JsonIgnore
	public DemoUserStore getStore() {
		return store;
	}

	private File getUserFile() throws IOException {
		File configDir = new File(getHomeDir(), PATH_CONFIG);
		FileTools.mkdirs(configDir);
		return new File(configDir, FILENAME_USER);
	}

	public synchronized DemoDoc importDocument(String name, ILocator locator) throws IOException {
		File repoDir = getRepoDir();
		FileTools.mkdirs(repoDir);

		purgeObsoleteDocuments();

		String uniqueName = makeUnique(name, n -> !documents.containsKey(n));
		// NOTE This might be overkill, but the file name could already be taken by an old file that
		// cannot be deleted.
		String uniqueFileName = makeUnique(name, fn -> !new File(repoDir, fn).exists());

		File file = new File(repoDir, uniqueFileName);
		LocatorTools.copy(locator, new FileLocator(file));

		DemoDoc document = new DemoDoc();
		document.setName(uniqueName);
		document.setPath(file.getCanonicalPath());
		documents.put(uniqueName, document);

		save();

		return document;
	}

	private synchronized void load() throws IOException {
		File userFile = getUserFile();
		getStore().load(this, userFile);
	}

	private String makeUnique(String name, Predicate<String> unique) {
		if (unique.test(name)) {
			return name;
		}

		String baseName = PathTools.getBaseName(name);
		String extension = PathTools.getExtension(name);
		int counter = 1;

		String uniqueName;
		do {
			uniqueName = String.format("%s-%d.%s", baseName, counter, extension);
			++counter;
		} while (!unique.test(uniqueName));

		return uniqueName;
	}

	private synchronized boolean purgeObsoleteDocuments() throws IOException {
		// Which files exist in the repository?
		Collection<File> existingFiles = new HashSet<>();
		File repoDir = getRepoDir();
		if (repoDir.exists()) {
			File[] files = repoDir.listFiles();
			if (files != null) {
				for (File file : files) {
					if (file.isFile()) {
						existingFiles.add(file.getCanonicalFile());
					}
				}
			}
		}

		// Synchronize the user's document list and the list of existing files and discard all
		// documents without a file.
		boolean anythingChanged = false;
		Iterator<DemoDoc> iterator = this.documents.values().iterator();
		while (iterator.hasNext()) {
			DemoDoc document = iterator.next();
			File file = new File(document.getPath()).getCanonicalFile();
			if (!existingFiles.remove(file)) {
				iterator.remove();
				anythingChanged = true;
			}
		}

		// Delete all files without a document.
		for (File obsoleteFile : existingFiles) {
			FileTools.delete(obsoleteFile);
		}

		return anythingChanged;
	}

	public DemoPreset resolvePreset() {
		DemoPreset preset = getPresets()
				.stream()
				.filter(each -> each.getId().equals(getSelectedPreset()))
				.findFirst()
				.orElseGet(() -> {
					if (StringTools.isEmpty(getSelectedPreset())) {
						setSelectedPreset("default");
					}
					return new DemoPreset(getSelectedPreset(), new HashMap<String, String>());
				});
		return preset;
	}

	public synchronized void save() throws IOException {
		File userFile = getUserFile();
		getStore().save(this, userFile);
	}

	public synchronized void setConfigurations(DtoConfigurations configurations) {
		if (this.configurations == null) {
			this.configurations = configurations;
		} else {
			this.configurations.merge(configurations);
		}
	}

	public synchronized void setDocuments(Map<String, DemoDoc> documents) {
		this.documents = documents;
	}

	public void setPresets(List<DemoPreset> presets) {
		this.presets = presets;
	}

	public void setSelectedPreset(String selectedPreset) {
		this.selectedPreset = selectedPreset;
	}

	public void setSettings(Map<String, String> settings) {
		this.settings = settings;
	}

}
