import { Injectable } from '@angular/core';
import { Observable } from 'rxjs';
import { tap } from 'rxjs/operators';

import { User } from '../model/user.model';
import { RemoteService } from '../service/remote.service';

@Injectable()
export class AuthService {

  constructor(
    private remoteService: RemoteService
  ) { }

  public user: User = new User();
  authModules: any[];

  /**
   * true if this "session" is authenticated.
   */
  isAuthenticated() {
    const token = this.getToken();
    return token;
  }

  /**
   * try to resume with initial browser URL.
   *
   * when entering the app from an authentication process, we expect the
   * new authentication token to be present.
   */
  resume() {
    const url = new URL(window.location.href);
    const searchParams = url.searchParams;
    const token = searchParams.get('auth_token');
    const state = searchParams.get('auth_state');
    if (token) {
      searchParams.delete('auth_token');
      searchParams.delete('auth_state');
      window.history.replaceState({}, 'resume', url.toString());
      if (state !== this.getState()) {
        return;
      }
      localStorage.setItem('auth_token', token);
    }
  }

  /**
   * enumerate the supported authentication modules
   */
  getAuthModules() {
    const urlTarget = this.remoteService.toTargetUrl('authentication/getModules');
    const request = {};
    return this.remoteService.fetch(urlTarget, request);
  }

  /**
   * we provide a simple authentication framework at the server.
   *
   * you need to provide the module you want to authenticate with. We expect an URL that represents the consent
   * gathering UI for the module.
   */
  request(module: string = "password") {
    const targetUrl = this.remoteService.toTargetUrl('authentication/request');
    let url;
    try {
      url = new URL(targetUrl);
    } catch(error) {
      url = new URL(this.remoteService.toExternalUrl(targetUrl));
    }
    const state = this.createState();
    const searchParams = url.searchParams;
    searchParams.set('auth_module', module);
    searchParams.set('auth_callback', this.remoteService.toExternalUrl('/app/documents'));
    searchParams.set('auth_state', state);
    window.location.href = url.toString();
  }

  /**
   * end authenticated session
   */
  logout(): Observable<void> {
    this.user = new User();
    localStorage.removeItem('auth_token');
    sessionStorage.removeItem('auth_state');
    const urlTarget = this.remoteService.toTargetUrl('authentication/logout');
    return this.remoteService
      .fetch(urlTarget, {});
  }

  /**
   * load the currently authenticated user
   */
  loadUser(): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('authentication/getUser');
    return this.remoteService
      .fetch(urlTarget, {})
      .pipe(
        tap((result) => {
          this.user = result.user;
        })
      );
  }

  /**
  * random local state to prevent CSRF upon authentication
  */
  protected getState(): string {
    return sessionStorage.getItem('auth_state');
  }

  /**
   * the currently associated authentication token
   */
  getToken() {
    return localStorage.getItem('auth_token');
  }

  /**
   * simple state creation
   */
  protected createState(): string {
    const state = this.createRandom(62);
    sessionStorage.setItem('auth_state', state);
    return state;
  }

  /**
   * simple random string creation
   */
  protected createRandom(length: number): string {
    let text = '';
    const possible = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    for (let i = 0; i < length; i++) {
      text += possible.charAt(Math.floor(Math.random() * possible.length));
    }
    return text;
  }

}
