import { Injectable } from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { Observable, Subject } from 'rxjs';

import { DemoService } from '../demo/demo.service';
import { PinEntryService } from '../pin-entry/pin-entry.service';
import { RemoteService } from '../service/remote.service';
import { ConversationMonitor } from './conversation-monitor.model';

declare var ConversationHandler: any;

@Injectable()
export class ConversationService {

  constructor(
    private demoService: DemoService,
    private pinEntryService: PinEntryService,
    private remoteService: RemoteService,
    private route: ActivatedRoute
  ) {
    this.init();
  }

  monitors: ConversationMonitor[] = [];

  monitorCreate$ = new Subject<ConversationMonitor>();

  monitorDestroy$ = new Subject<ConversationMonitor>();

  monitorActive = 0;

  init(): void {
    const that = this;
    const handler = new ConversationHandler();
    /**
     * we want to dispatch the acknowledge via our backend.
     *
     * this ensures we dont need to stream result down to client.
     * in order to do this, we need to use our own http client, adding
     * an authentication token.
     */
    handler.urlAcknowledge = this.remoteService.toTargetUrl('demo/conversation/acknowledge');
    handler.call = (url, request) => {
      return that.remoteService.fetch(url, request).toPromise();
    };
    handler.onStageRequestInput = this.pinEntryService.requestInput.bind(this.pinEntryService);
    ConversationHandler.active = handler;
  }

  createMonitor(conversation: string) {
    const monitor = new ConversationMonitor(this);
    monitor.conversation = conversation;
    this.monitors = [...this.monitors, monitor];
    this.monitorActive = this.monitorActive + 1;
    this.monitorCreate$.next(monitor);
    return monitor;
  }

  destroyMonitor(monitor: ConversationMonitor) {
    this.monitorActive = this.monitorActive - 1;
    this.monitors = this.monitors.filter((it) => it !== monitor);
    this.monitorDestroy$.next(monitor);
  }

  public resume() {
    const conversation = this.route.snapshot.queryParamMap.get('cs_conversation');
    if (!conversation) {
      return;
    }
    const conversationMonitor = this.createMonitor(conversation);
    conversationMonitor.begin('Resume...');
    const context = {
      monitor: conversationMonitor
    };
    ConversationHandler.active
      .resumeAfterLoad(context)
      .then(() => {
        conversationMonitor.end();
        this.demoService.loadDocuments().subscribe({
          next: () => { },
          error: () => { }
        });
      })
      .catch((e) => {
        conversationMonitor.end();
        throw e;
      });
  }

  public cancel(conversation: string): Observable<any> {
    const urlTarget = this.remoteService.toTargetUrl('demo/conversation/cancel');
    const request = {
      conversation
    };
    return this.remoteService.fetch(urlTarget, request);
  }

}
