import { NgIf } from '@angular/common';
import { Component, EmbeddedViewRef, Input, OnDestroy, OnInit, TemplateRef, ViewChild, ViewContainerRef } from '@angular/core';
import { MatIconButton } from '@angular/material/button';
import { MatCard, MatCardContent } from '@angular/material/card';
import { MatIcon } from '@angular/material/icon';
import { MatProgressBar } from '@angular/material/progress-bar';
import { Subject } from 'rxjs';
import { delay, takeUntil } from 'rxjs/operators';

import { ConversationMonitor } from './conversation-monitor.model';
import { ConversationService } from './conversation.service';

@Component({
    selector: 'app-conversation-monitor',
    imports: [MatCard, MatCardContent, MatIconButton, MatIcon, NgIf, MatProgressBar],
    templateUrl: './conversation-monitor.component.html',
    styleUrls: ['./conversation-monitor.component.scss']
})
export class ConversationMonitorComponent implements OnInit, OnDestroy {

  @Input() createDelay = 1000;

  @Input() destroyDelay = 3000;

  @ViewChild('monitors', { read: ViewContainerRef, static: true }) monitorContainer: ViewContainerRef;

  @ViewChild('monitor', { static: true }) monitorTemplate: TemplateRef<any>;

  private destroyed$ = new Subject<boolean>();

  constructor(
    private conversationService: ConversationService
  ) {
  }

  ngOnInit() {
    this.conversationService.monitors.forEach(monitor => {
      this.create(monitor);
    });
    this.conversationService.monitorCreate$
      .pipe(
        takeUntil(this.destroyed$),
        delay(this.createDelay)
      )
      .subscribe((monitor) => this.create(<ConversationMonitor>monitor));
    this.conversationService.monitorDestroy$
      .pipe(
        takeUntil(this.destroyed$),
        delay(this.destroyDelay)
      )
      .subscribe((monitor) => this.destroy(monitor));
  }

  ngOnDestroy() {
    this.destroyed$.next(undefined);
    this.destroyed$.complete();
  }

  isActive() {
    return this.conversationService.monitorActive > 0;
  }

  private create(monitor: ConversationMonitor) {
    if (monitor.isAtEnd()) {
      return;
    }
    const context = {
      model: monitor
    };
    const monitorView = this.monitorTemplate.createEmbeddedView(context);
    this.monitorContainer.insert(monitorView);
  }

  private destroy(monitor: ConversationMonitor) {
    for (let i = 0; i < this.monitorContainer.length; i++) {
      const ref = <EmbeddedViewRef<any>>this.monitorContainer.get(i);
      if (ref.context.model === monitor) {
        ref.destroy();
      }
    }
  }

  onCancel(monitor: ConversationMonitor): void {
    monitor.cancel();
  }

}
