import { NgIf } from '@angular/common';
import { AfterViewInit, Component, OnInit } from '@angular/core';
import { MatButton, MatIconButton } from '@angular/material/button';
import { MatIcon } from '@angular/material/icon';
import { MatMenu, MatMenuItem, MatMenuTrigger } from '@angular/material/menu';
import { MatToolbar } from '@angular/material/toolbar';
import { Router, RouterLink, RouterOutlet } from '@angular/router';

import { AuthService } from '../auth/auth.service';
import { SettingsService } from '../settings/settings.service';
import { BusyIndicator } from '../shared/busy/busy-indicator.service';
import { DemoService } from './demo.service';

import * as metainfo from '../meta.json';

@Component({
    selector: 'app-demo',
    imports: [MatToolbar, NgIf, MatButton, MatIconButton, RouterLink, MatIcon, MatMenuTrigger, MatMenu, MatMenuItem, RouterOutlet],
    templateUrl: './demo.component.html',
    styleUrls: ['./demo.component.scss']
})
export class DemoComponent implements AfterViewInit, OnInit {

  public metainfo = metainfo;

  constructor(
    private router: Router,
    private authService: AuthService,
    private demoService: DemoService,
    private settingsService: SettingsService
  ) {
    console.log(`${this.metainfo.title} v.${this.metainfo.version}, Build ${this.metainfo.build}, ${this.metainfo.timestamp}`);
  }

  loading = new BusyIndicator();

  ngAfterViewInit() {
    if (this.isAuthenticated()) {
      this.router.navigate(['/app/documents']);
    }
  }

  ngOnInit() {
    this.authService.resume();
    this.authService.loadUser().subscribe();
  }

  isActive(element): boolean {
    return element.classList.contains('router-link-active');
  }

  isAuthenticated() {
    return this.authService.isAuthenticated();
  }

  getUserName() {
    return this.authService.user.name;
  }

  onLogout() {
    this.authService.logout().subscribe();
    this.router.navigate(['/app']);
  }

  onLogon() {
    this.loading.do(() => this.authService.getAuthModules().subscribe((result) => {
      if (result.length == 0) {
        throw "No authentication modules available.";
      } else {
        this.authService.request(result[0].id);
      }
    }));
  }

  /**
   * Reset all state associated with the user, which is currently logged in. Used by end-to-end tests.
   *
   * @param callback called on success
   */
  resetUserState() {
    this.settingsService.save([]).subscribe();
    this.demoService.removeAllDocuments().subscribe();
  }
}
