import { Injectable } from '@angular/core';

import { Observable, BehaviorSubject, from } from 'rxjs';
import { tap, mergeMap } from 'rxjs/operators';

import { Document } from '../model/document.model';
import { RemoteService } from '../service/remote.service';
import { ConversationMonitor } from '../conversation/conversation-monitor.model';

declare var ConversationHandler: any;

@Injectable()
export class DemoService {

  private document$ = new BehaviorSubject<Document[]>([]);

  constructor(
    private remoteService: RemoteService
  ) {
  }

  public isEmpty() {
    const documents = this.document$.getValue();
    return documents.length === 0;
  }

  getDocuments(): Observable<Document[]> {
    return this.document$;
  }

  loadDocuments(): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/getDocuments');
    const request = {
    };
    return this.remoteService //
      .fetch(urlTarget, request, { message: 'Loading...' })
      .pipe(
        tap(result => this.document$.next(result))
      );
  }

  removeAllDocuments(): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/deleteAllDocuments');
    return this.remoteService
      .fetch(urlTarget, null)
      .pipe(
        mergeMap(() => this.loadDocuments())
      );
  }

  removeDocument(doc: Document): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/deleteDocument');
    const request = {
      document: doc
    };
    return this.remoteService
      .fetch(urlTarget, request, { message: 'Removing...' })
      .pipe(
        mergeMap(() => this.loadDocuments())
      );
  }

  removeDocuments(docs: Document[]): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/deleteDocuments');
    const request = {
      documents: docs
    };
    return this.remoteService
      .fetch(urlTarget, request, { message: 'Removing...' })
      .pipe(
        mergeMap(() => this.loadDocuments())
      );
  }

  downloadDocument(doc: Document): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/downloadDocument');
    const request = {
      document: doc
    };
    return from(this.remoteService.download(urlTarget, request, { message: 'Download...' }));
  }

  downloadDocuments(docs: Document[]): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/downloadDocuments');
    const request = {
      documents: docs
    };
    return from(this.remoteService.download(urlTarget, request, { message: 'Download...' }));
  }

  openDocument(doc: Document, monitor: ConversationMonitor = null): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/openDocument');
    const request = {
      document: doc,
      redirectUri: window.location.href
    };
    return this.remoteService
      .fetchConversational(urlTarget, request, { message: 'Start viewer...' }, { monitor: monitor })
      .pipe(
        mergeMap(() => this.loadDocuments())
      );
  }
  editDocument(doc: Document): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/editDocument');
    const request = {
      document: doc,
      redirectUri: window.location.href
    };
      return this.remoteService
      .fetch(urlTarget, request, { message: 'Saving...' });
  }

  embedDocument(doc: Document): Observable<any> {
    const urlTarget = this.remoteService.toTargetUrl('demo/openDocument');
    const request = {
      document: doc
    };
    return this.remoteService
      .fetchSnapshot(urlTarget, request, { message: 'Start viewer...' });
  }

  handleSnapshot(snapshot: any): Observable<void> {
    throw new Error('Method not implemented.');
  }

  openDocuments(docs: Document[], monitor: ConversationMonitor = null): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/openDocuments');
    const request = {
      documents: docs,
      redirectUri: window.location.href
    };
    return this.remoteService
      .fetchConversational(urlTarget, request, { message: 'Start explorer...' }, { monitor: monitor })
      .pipe(
        mergeMap(() => this.loadDocuments())
      );
  }

  timestampDocument(doc: Document, monitor: ConversationMonitor = null): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/timestampDocument');
    const request = {
      document: doc,
      redirectUri: window.location.href
    };
    if (monitor) {
      monitor.subTaskName = 'Send request...';
    }
    return this.remoteService
      .fetchConversational(urlTarget, request, { message: 'Timestamp...' }, { monitor: monitor })
      .pipe(
        mergeMap((result) => this.loadDocuments())
      );
  }

  timestampDocuments(docs: Document[], monitor: ConversationMonitor = null): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/timestampDocuments');
    const request = {
      documents: docs,
      redirectUri: window.location.href
    };
    if (monitor) {
      monitor.subTaskName = 'Send request...';
    }
    return this.remoteService
      .fetchConversational(urlTarget, request, { message: 'Timestamp...' }, { monitor: monitor })
      .pipe(
        mergeMap((result) => this.loadDocuments())
      );
  }

  signDocument(doc: Document, monitor: ConversationMonitor = null): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/signDocument');
    const request = {
      document: doc,
      redirectUri: window.location.href
    };
    if (monitor) {
      monitor.subTaskName = 'Send request...';
    }
    return this.remoteService
      .fetchConversational(urlTarget, request, { message: 'Sign...' }, { monitor: monitor })
      .pipe(
        mergeMap((result) => this.loadDocuments())
      );
  }

  signDocuments(docs: Document[], monitor: ConversationMonitor = null): Observable<void> {
    const urlTarget = this.remoteService.toTargetUrl('demo/signDocuments');
    const request = {
      documents: docs,
      redirectUri: window.location.href
    };
    if (monitor) {
      monitor.subTaskName = 'Send request...';
    }
    return this.remoteService
      .fetchConversational(urlTarget, request, { message: 'Sign...' }, { monitor: monitor })
      .pipe(
        mergeMap((result) => this.loadDocuments())
      );
  }
}
