import { SelectionModel } from '@angular/cdk/collections';
import { NgClass } from '@angular/common';
import { HttpClient } from '@angular/common/http';
import { Component, ElementRef, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { MatButton } from '@angular/material/button';
import { MatCheckbox } from '@angular/material/checkbox';
import { MatDialog } from '@angular/material/dialog';
import { MatIcon } from '@angular/material/icon';
import { MatPaginator } from '@angular/material/paginator';
import { MatCell, MatCellDef, MatColumnDef, MatHeaderCell, MatHeaderCellDef, MatHeaderRow, MatHeaderRowDef, MatRow, MatRowDef, MatTable, MatTableDataSource } from '@angular/material/table';
import { MatTooltip } from '@angular/material/tooltip';
import { Router } from '@angular/router';
import { BlockUI, NgBlockUI } from 'ng-block-ui';
import { FileUploader, FileUploadModule } from 'ng2-file-upload';
import { Subject } from 'rxjs';
import { finalize } from 'rxjs/operators';

import { AuthService } from '../auth/auth.service';
import { ConversationMonitorComponent } from '../conversation/conversation-monitor.component';
import { ConversationService } from '../conversation/conversation.service';
import { DemoService } from '../demo/demo.service';
import { Document } from '../model/document.model';
import { NoticeService } from '../notice/notice.service';
import { FileSizePipe } from '../pipe/file-size.pipe';
import { RemoteService } from '../service/remote.service';
import { SettingsService } from '../settings/settings.service';
import { DocumentSettingsComponent } from './document-settings.component';

const initialSelection = [];
const allowMultiSelect = true;

@Component({
    selector: 'app-documents',
    imports: [ConversationMonitorComponent, MatCheckbox, MatButton, NgClass, MatIcon, FileUploadModule, MatTable, MatColumnDef, MatHeaderCellDef, MatHeaderCell, MatCellDef, MatCell, MatTooltip, MatHeaderRowDef, MatHeaderRow, MatRowDef, MatRow, MatPaginator, FileSizePipe],
    templateUrl: './documents.component.html',
    styleUrls: ['./documents.component.scss']
})
export class DocumentsComponent implements OnInit, OnDestroy {

  uploader: FileUploader;

  public dropActive = false;

  public selection: SelectionModel<Document> = new SelectionModel<Document>(allowMultiSelect, initialSelection);

  dataSource: MatTableDataSource<Document>;

  displayedColumns = ['select', 'name', 'length', 'state', 'action'];

  @BlockUI() blockUI: NgBlockUI;

  @ViewChild('fileSelector', { static: true }) fileSelector: ElementRef;
  @ViewChild(MatPaginator, { static: true }) paginator: MatPaginator;

  private destroyed$ = new Subject<boolean>();

  constructor(
    private conversationService: ConversationService,
    private authService: AuthService,
    private noticeService: NoticeService,
    private remoteService: RemoteService,
    private settingsService: SettingsService,
    private demoService: DemoService,
    private router: Router,
    private dialog: MatDialog,
    private httpClient: HttpClient) {
  }

  ngOnInit(): void {
    this.conversationService.resume();
    this.uploader = new FileUploader({
      url: this.remoteService.toTargetUrl('demo/uploadDocument'),
      autoUpload: true,
      headers: [{
        name: 'Authorization',
        value: 'Bearer ' + this.authService.getToken()
      }]
    });
    this.uploader.onAfterAddingAll = () => {
      this.blockUI.reset();
      this.blockUI.start('Upload...');
    };
    this.uploader.onBeforeUploadItem = (item) => {
      // settings may have changed
      const urlTarget = this.remoteService.toTargetUrl('demo/uploadDocument');
      this.uploader.setOptions({ removeAfterUpload: true, url: urlTarget });
      item.withCredentials = true;
    };
    this.uploader.onProgressItem = (item, progress) => {
      this.blockUI.update('Upload ' + item.file.name + ' ' + progress + '%');
    };
    this.uploader.onCompleteAll = () => {
      this.blockUI.stop();
      this.selection.clear();
      this.noticeService.info('Upload complete', 'Service info');
      this.demoService.loadDocuments().subscribe();
    };
    this.uploader.onErrorItem = () => {
      this.blockUI.stop();
      this.selection.clear();
      this.demoService.loadDocuments().subscribe();
    };
    this.dataSource = new MatTableDataSource([]);
    this.dataSource.paginator = this.paginator;
    this.demoService.getDocuments().subscribe((documents) => this.dataSource.data = documents);
    this.demoService.loadDocuments().subscribe();
  }

  ngOnDestroy() {
    this.destroyed$.next(undefined);
    this.destroyed$.complete();
  }

  public fileOverBase(e: any): void {
    this.dropActive = e;
  }

  public isViewed(row: Document) {
    if (row == null) {
      return false;
    }
    return row.viewed;
  }

  public isSigned(row: Document) {
    if (row == null) {
      return false;
    }
    return row.signed;
  }

  public isTimestamped(row: Document) {
    if (row == null) {
      return false;
    }
    return row.timestamped;
  }

  public isEmpty() {
    return this.demoService.isEmpty();
  }

  public clickFileSelector(event: MouseEvent) {
    if (event.ctrlKey) {
      this.httpClient.get(this.remoteService.toExternalUrl('/assets/empty.pdf'), { responseType: 'blob' }).subscribe(
        (response) => {
          let name = 'Built-in PDF document (2 pages)';
          this.uploader.addToQueue([new File([response], name + ".pdf")]);
          this.uploader.uploadAll();
        }
      );
    } else {
      this.fileSelector.nativeElement.click();
    }
  }

  public loadDocuments() {
    this.selection.clear();
    this.demoService.loadDocuments().subscribe();
  }

  public removeDocument(doc: Document) {
    this.selection.clear();
    this.demoService.removeDocument(doc).subscribe();
  }

  public removeDocuments() {
    const tmpDocuments = this.selection.selected;
    this.selection.clear();
    this.demoService.removeDocuments(tmpDocuments).subscribe();
  }

  public signDocument(doc: Document) {
    const conversationMonitor = this.conversationService.createMonitor(null);
    conversationMonitor.begin('Sign document...');
    this.demoService.signDocument(doc, conversationMonitor)
      .pipe(
        finalize(() => {
          conversationMonitor.end();
        })
      )
      // force events
      .subscribe();
  }

  public signDocuments() {
    const tmpDocuments = this.selection.selected;
    this.selection.clear();
    const conversationMonitor = this.conversationService.createMonitor(null);
    conversationMonitor.begin('Sign documents...');
    this.demoService.signDocuments(tmpDocuments, conversationMonitor)
      .pipe(
        finalize(() => {
          conversationMonitor.end();
        })
      )
      // force events
      .subscribe();
  }


  public timestampDocument(doc: Document) {
    const conversationMonitor = this.conversationService.createMonitor(null);
    conversationMonitor.begin('Timestamp document...');
    this.demoService.timestampDocument(doc, conversationMonitor)
      .pipe(
        finalize(() => {
          conversationMonitor.end();
        })
      )
      // force events
      .subscribe();
  }

  public timestampDocuments() {
    const tmpDocuments = this.selection.selected;
    this.selection.clear();
    const conversationMonitor = this.conversationService.createMonitor(null);
    conversationMonitor.begin('Timestamp documents...');
    this.demoService.timestampDocuments(tmpDocuments, conversationMonitor)
      .pipe(
        finalize(() => {
          conversationMonitor.end();
        })
      )
      // force events
      .subscribe();
  }

  public openDocument(doc: Document, event: MouseEvent) {
    const conversationMonitor = this.conversationService.createMonitor(null);
    conversationMonitor.begin('View document...');
    if (event.ctrlKey) {
      this.demoService
        .embedDocument(doc)
        .pipe(
          finalize(() => {
            conversationMonitor.end();
          })
        )
        .subscribe((snapshot) => {
          if (snapshot.stage.scheme === 'urn:intarsys:names:conversation:1.0:schemes:HttpRedirect') {
            this.router.navigate(['/', 'embedded'], { queryParams: { conversation: snapshot.conversation, url: snapshot.stage.url } });
          } else {
            throw new Error('unexpected scheme ' + snapshot.stage.scheme);
          }
        });
    } else {
      this.demoService.openDocument(doc, conversationMonitor)
        .pipe(
          finalize(() => {
            conversationMonitor.end();
          })
        )
        .subscribe();
    }
  }

  public openDocumentSettings(doc: Document) {
    const dialogRef = this.dialog.open(DocumentSettingsComponent, {
      minWidth: '50vw',
      data: { docProperties: doc.properties['properties'] }
    });
    dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        doc.properties['properties'] = result;
        this.demoService.editDocument(doc)
          .subscribe(() => {
          });
      }
    });
  }

  public openDocuments() {
    const conversationMonitor = this.conversationService.createMonitor(null);
    conversationMonitor.begin('View documents...');
    this.demoService.openDocuments(this.selection.selected, conversationMonitor)
      .pipe(
        finalize(() => {
          conversationMonitor.end();
        })
      )
      .subscribe();
  }

  public downloadDocument(doc: Document) {
    this.demoService
      .downloadDocument(doc)
      .subscribe();
  }


  public downloadDocuments() {
    const tmpDocuments = this.selection.selected;
    this.selection.clear();
    this.demoService
      .downloadDocuments(tmpDocuments)
      .pipe(
        finalize(() => this.selection.clear())
      )
      .subscribe();
  }

  public selectionEmpty() {
    return this.selection.isEmpty();
  }

  isAllSelected() {
    const numSelected = this.selection.selected.length;
    const numRows = this.dataSource.data.length;
    return numSelected === numRows;
  }

  masterToggle() {
    this.isAllSelected() ?
      this.selection.clear() :
      this.dataSource.data.forEach(row => this.selection.select(row));
  }
}
