import { Injectable, NgZone } from '@angular/core';
import { MatSnackBar } from '@angular/material/snack-bar';
import { Observable, Subject } from 'rxjs';

import { Notice } from './notice.model';

@Injectable()
export class NoticeService {

  private subject = new Subject<Notice>();

  constructor(
    private snackBar: MatSnackBar,
    private zone: NgZone
  ) {
    this.init();
  }

  getNotice(): Observable<any> {
    return this.subject.asObservable();
  }

  error(message: string, title = 'Error', sticky = false) {
    this.notice('error', message, title, sticky);
  }

  info(message: string, title = 'Information', sticky = false) {
    this.notice('info', message, title, sticky);
  }

  warning(message: string, title = 'Warning', sticky = false) {
    this.notice('warning', message, title, sticky);
  }

  notice(type: string, message: string, title: string = 'Notice', sticky = false) {
    if (message) {
      message = message.replace('<', () => '&lt;');
    }
    this.subject.next(<Notice>{ type: type, title: title, message: message, sticky: sticky });
  }

  clear() {
    this.subject.next(undefined);
  }

  init(): void {
    this.getNotice().subscribe((notice: Notice) => {
      if (!notice) {
        return;
      }
      let options;
      let type;
      if ('info' === notice.type) {
        type = 'info';
        options = {
          duration: 5000
        };
      } else if ('warning' === notice.type) {
        type = 'warn';
        options = {
          duration: 5000
        };
      } else if ('error' === notice.type) {
        type = 'error';
        options = {
          duration: 0
        };
      } else {
        type = 'info';
        options = {
          duration: 5000
        };
      }
      if (notice.sticky) {
        options.duration = 0;
      }
      options.horizontalPosition = 'end';
      let message = notice.message;
      if (notice.title) {
        message = `${notice.title}: ${message}`;
      }
      const snackBarRef = this.zone.run(() => this.snackBar.open(message, 'Close', options));
      snackBarRef.onAction().subscribe(() => this.zone.run(() => snackBarRef.dismiss()));
    });
  }
}
