import { NgFor, NgIf } from '@angular/common';
import { Component, OnInit } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatIconButton } from '@angular/material/button';
import { MatButtonToggle, MatButtonToggleGroup } from '@angular/material/button-toggle';
import { MatCard, MatCardContent } from '@angular/material/card';
import { MatOption } from '@angular/material/core';
import { MatFormField, MatLabel } from '@angular/material/form-field';
import { MatIcon } from '@angular/material/icon';
import { MatInput } from '@angular/material/input';
import { MatSelect } from '@angular/material/select';
import { MatTooltip } from '@angular/material/tooltip';

import { saveAs } from 'file-saver';
import { catchError, forkJoin, map, mergeMap, Observable, throwError } from 'rxjs';
import { NoticeService } from '../notice/notice.service';
import { TextareaResizeDirective } from '../shared/textarea/textarea-resize.directive';
import { readAsText } from '../shared/util/blobs';
import { SettingsService } from './settings.service';


function readPresets(file: File): Observable<any[]> {
  return readAsText(file).pipe(
    map(text => JSON.parse(text)),
    map(toPresets),
    catchError(cause => throwError(() => new Error(`Invalid file ${file.name}: ${cause.message}`, { cause })))
  );
}

function mergePresets(presets: any[][]): any[] {
  const map = new Map<string, any>();
  for (const preset of presets.flat()) {
    map.set(preset.id, preset);
  }

  return Array.from(map.values());
}

function toPresets(input: any): any[] {
  if (isPreset(input)) {
    return [input];
  }

  if (Array.isArray(input) && input.every(isPreset)) {
      return input;
  }

  throw new Error('Wrong schema');
}

// Unfortunately, there is no type definition for presets. They are treated as any everywhere.
// This is only a shallow test. The backend might still raise a deserialization error.
function isPreset(candidate: any): boolean {
  return typeof candidate == 'object'
    && candidate !== null
    && Object.prototype.hasOwnProperty.call(candidate, 'id')
    && typeof candidate.id === 'string';
}

function toErrorMessage(error: any): string {
  if (error instanceof Error) {
    return error.message;
  }

  return typeof error === 'string'
      ? error
      : 'Unknown error';
}

@Component({
    selector: 'app-settings',
    imports: [MatFormField, MatLabel, MatSelect, NgFor, MatOption, MatIconButton, MatTooltip, MatIcon, FormsModule, MatInput, MatButtonToggleGroup, MatButtonToggle, NgIf, MatCard, MatCardContent, TextareaResizeDirective],
    templateUrl: './settings.component.html',
    styleUrls: ['./settings.component.css']
})
export class SettingsComponent implements OnInit {

  constructor(
    private settingsService: SettingsService,
    private noticeService: NoticeService
  ) {}

  presets: any[] = [];
  preset: any = {};

  private updateFromService() {
    this.presets = this.settingsService.presets;
    this.preset = this.settingsService.preset;
  }

  ngOnInit() {
    this.updateFromService();
  }

  getPresetValues(): any[] {
    // avoid using "values" (not present in IE) instead of adding polyfill
    return Object.keys(this.presets).map(key => this.presets[key]).sort((a,b) => (a.id < b.id ? -1 : 1));
  }

  isBasicAuth() {
    return this.preset.settings['gears.core.auth.type'] === "basic";
  }

  onSelectPreset() {
    this.settingsService
      .setPresetId(this.preset.id)
      .subscribe(() => this.updateFromService());
  }

  onRemovePreset() {
    const input = confirm("Remove current preset?");
    if (!input) {
      return;
    }
    this.settingsService
      .removePreset(this.preset)
      .subscribe(() => this.updateFromService());
  }

  onCreatePreset() {
    const input = prompt("Enter a name for the new preset", this.preset?.id);
    if (!input || input === this.preset?.id) {
      return;
    }
    this.settingsService
      .createPreset(input)
      .subscribe(() => this.updateFromService());
  }

  onExportPresets() {
    const jsonPresets = JSON.stringify(this.presets, null, 2);
    saveAs(new Blob([jsonPresets], { type: 'application/json' }), 'gears-presets.json');
  }

  onImportPresets(fileList: FileList): void {
    if (fileList.length === 0) {
      return;
    }

    const files = Array.from(fileList)
    forkJoin(files.map(file => readPresets(file)))
        .pipe(
          map(presets => mergePresets([this.presets, ...presets])),
          mergeMap(presets => this.settingsService.save(presets))
        )
			.subscribe({
				next: () => this.updateFromService(),
        error: error => this.noticeService.error(toErrorMessage(error), 'Import error')
			});
  }

  submit(): void {
    this.settingsService
      .save(this.presets)
      .subscribe();
  }
}
