const SCHEME_Cancel = 'urn:intarsys:names:conversation:1.0:schemes:Cancel';
const SCHEME_Error = 'urn:intarsys:names:conversation:1.0:schemes:Error';
const SCHEME_HttpRedirect = 'urn:intarsys:names:conversation:1.0:schemes:HttpRedirect';
const SCHEME_Idle = 'urn:intarsys:names:conversation:1.0:schemes:Idle';
const SCHEME_Processing = 'urn:intarsys:names:conversation:1.0:schemes:Processing';
const SCHEME_RequestInput = 'urn:intarsys:names:conversation:1.0:schemes:RequestInput';
const SCHEME_Result = 'urn:intarsys:names:conversation:1.0:schemes:Result';

const ERR_INVALID_ARGUMENT = "Conversation.InvalidArgument";
const ERR_CONFIGURATION_ERROR = "Conversation.ConfigurationError";
const ERR_PROTOCOL_ERROR = "Conversation.ProtocolError";
const ERR_ERROR_STAGE = "Conversation.ErrorStage";
const ERR_CANCEL_STAGE = "Conversation.CancelStage";

const DEFAULT_STICKY_FIELD_NAME = "sticky_id";

function ConversationError(conversation, code, message) {
  this.conversation = conversation;
  this.code = code;
  this.message = message;
  // this.stack = (new Error()).stack;
}
ConversationError.prototype = Object.create(Error.prototype);
ConversationError.prototype.constructor = ConversationError;
ConversationError.prototype.name = 'ConversationError';

function ConversationHandler() {
  var that = this;
  this.urlAcknowledge = null;
  this.stickyFieldName = DEFAULT_STICKY_FIELD_NAME;
  this.acknowledge = function (request, context) {
    if (!context) {
      context = {};
    }
    var url = that.urlAcknowledge;
    if (context.stickyId) {
      url = new URI(that.urlAcknowledge).setSearch(this.stickyFieldName, context.stickyId).toString();
    }
    return this.call(url, request);
  };
  this.call = function (url, request) {
    return callJson(url, request);
  };
}
ConversationHandler.prototype = {};
ConversationHandler.prototype.constructor = ConversationHandler;

ConversationHandler.prototype.onStageRequestInput = function (stage) {
  return Promise.resolve(null);
}

/*
 * This function should be called whenever we get a conversational result from a service.
 *
 * It handles all the low level mechanics of the conversation.
 *
 * @param snapshot
 *    is the result of the last conversational call.
 * @param context
 *    holds contextual properties for tweaking the handle behavior
 */
ConversationHandler.prototype.handle = function (snapshot, context, deferred) {
  if (snapshot === undefined || snapshot == null) {
    return Promise.reject(new ConversationError(null, ERR_INVALID_ARGUMENT, "argument 'snapshot' missing"));
  }
  if (snapshot.conversation === null) {
    return Promise.reject(new ConversationError(null, ERR_INVALID_ARGUMENT, "argument 'snapshot.conversation' missing"));
  }
  if (snapshot.stage === null) {
    return Promise.reject(new ConversationError(null, ERR_INVALID_ARGUMENT, "argument 'snapshot.stage' missing"));
  }
  var that = this;
  context = context || {};
  if (context.monitor) {
    context.monitor.update(snapshot);
  }
  if (context.handleIntercept) {
    var handled = context.handleIntercept(snapshot, context, deferred);
    if (handled) {
      return handled;
    }
  }
  var logPrefix = "cs: handle conversation " + snapshot.conversation + ", stage " + snapshot.stage.id + ": ";
  if (SCHEME_HttpRedirect === snapshot.stage.scheme) {
    var redirectUrl = new URI(snapshot.stage.url);
    if (snapshot.stage.outOfBand) {
      var name = "_blank";
      var specs = "location=no,menubar=no,status=no,toolbar=no,width=800,height=400";
      if (snapshot.stage.name) {
        name = snapshot.stage.name;
      }
      if (snapshot.stage.specs || snapshot.stage.specs == "") {
        specs = snapshot.stage.specs;
      }
      var childWindow = window.open(redirectUrl.toString(), name, specs);
      if (childWindow) {
        try {
          if (snapshot.stage.print) {
            childWindow.onload = function () {
              childWindow.focus();
              childWindow.onafterprint = function () {
                childWindow.close();
              };
              childWindow.print();
            };
          }
        } catch (e) {
          // this may throw a CORS exception
          console.warn("printing impossible due to CORS restrictions")
        }
      }
      return that.resumeSnapshot(snapshot, context, deferred, /* value */ null);
    } else {
      var search = redirectUrl.search(true);
      if (search["redirectUri"] == "?") {
        var callbackUrl = new URI(context.callback == null ? window.location.href : context.callback);
        if (context.queryParams) {
          for (var prop in context.queryParams) {
            callbackUrl.setSearch(prop, context.queryParams[prop]);
          }
        }
        redirectUrl.setSearch("redirectUri", callbackUrl.toString());
      }
      if (snapshot.stage.name) {
        var name = snapshot.stage.name;
        var specs = "location=no,menubar=no,status=no,toolbar=no,width=800,height=400";
        if (snapshot.stage.specs || snapshot.stage.specs == "") {
          specs = snapshot.stage.specs;
        }
        window.open(redirectUrl.toString(), name, specs);
      } else {
        window.location = redirectUrl.toString();
      }
      return new Promise(function (resolve, reject) {
        // some promises are never fullfilled...
      });
    }
  } else if (SCHEME_Processing === snapshot.stage.scheme) {
    var tmpDeferred;
    if (deferred) {
      tmpDeferred = deferred;
    } else {
      tmpDeferred = {};
      tmpDeferred.promise = new Promise(function (resolve, reject) {
        tmpDeferred.resolve = resolve;
        tmpDeferred.reject = reject;
      });
    }
    setTimeout(function () {
      that //
        .resumeSnapshot(snapshot, context, tmpDeferred, /* value */ null) //
        .then(function (result) {
          tmpDeferred.resolve(result);
        }, function (error) {
          tmpDeferred.reject(error);
        });
    }, 500);
    return tmpDeferred.promise;
  } else if (SCHEME_RequestInput === snapshot.stage.scheme) {
    return that
      .onStageRequestInput(snapshot.stage)
      .then(function (value) {
        return that.resumeSnapshot(snapshot, context, deferred, value);
      }, function (error) {
        return Promise.reject(error);
      });
  } else if (SCHEME_Idle === snapshot.stage.scheme) {
    return Promise.resolve(null);
  }
  if (SCHEME_Result === snapshot.stage.scheme) {
    return Promise.resolve(snapshot.stage.result);
  } else if (SCHEME_Error === snapshot.stage.scheme) {
    var errorResponse = snapshot.stage.error;
    if (errorResponse === null) {
      errorResponse = {
        code: "unknown",
        message: "conversation failed"
      }
    }
    return Promise.reject(new ConversationError(snapshot.conversation, ERR_ERROR_STAGE + "." + errorResponse.code, errorResponse.message));
  } else if (SCHEME_Cancel === snapshot.stage.scheme) {
    return Promise.reject(new ConversationError(snapshot.conversation, ERR_CANCEL_STAGE, "conversation canceled"));
  } else {
    console.warn(logPrefix + "unsupported scheme " + snapshot.stage.scheme);
    return Promise.reject(new ConversationError(snapshot.conversation, ERR_INVALID_ARGUMENT, "argument 'snapshot.stage.scheme=" + snapshot.stage.scheme + "' not supported"));
  }
}

/**
 * Acknowledge a snapshot and handle the result.
 *
 * @param {The snapshot to be acknowledged} snapshot
 * @param {An optional execution context} context
 * @param {A promise for handling deferred resolution} deferred
 * @param {An optional acknowledge value} value
 * @returns
 */
ConversationHandler.prototype.resumeSnapshot = function (snapshot, context, deferred, value) {
  context = context || {};
  if (value === undefined) {
    value = null;
  }
  var that = this;
  var request = {
    conversation: snapshot.conversation,
    inReplyTo: snapshot.stage == null ? null : snapshot.stage.id
  };
  if (value != null) {
    request.value = value;
  }
  return this.acknowledge(request, context).then(function (response) {
    return that.handle(response.snapshot, context, deferred);
  }, function (error) {
    return Promise.reject(new ConversationError(snapshot.conversation, ERR_PROTOCOL_ERROR, "" + error.code + " - " + error.message));
  });
}

/*
 * This function should be called "onLoad" of the client application dealing
 * with conversations.
 *
 * It ensures that after a redirect, the client tries to resume a conversation
 * designated in the cs_conversation parameter.
 */
ConversationHandler.prototype.resumeAfterLoad = function (context) {
  context = context || {};
  //
  var uri = URI(window.location.href);
  var query = uri.search(true);
  var conversation = query["cs_conversation"];
  var stage = query["cs_stage"];
  var outcome = query["cs_outcome"];
  context.stickyId = null;
  if (this.stickyFieldName) {
    context.stickyId = query[this.stickyFieldName];
  }
  uri.removeSearch("cs_conversation");
  uri.removeSearch("cs_stage");
  uri.removeSearch("cs_outcome");
  if (conversation == null || conversation === '') {
    return Promise.resolve(null);
  }
  window.history.replaceState({}, "conversation resumed", uri.toString());
  var snapshot = {
    conversation: conversation,
  };
  if (outcome) {
    /**
     * when we have an outcome, backend resources are already disposed
     * and acknowledge is skipped.
     */
    try {
      var decoded = atob(outcome);
      snapshot.stage = JSON.parse(decoded);
    } catch (e) {
      snapshot.stage = {
        scheme: SCHEME_Error,
        error: {
          code: ERR_PROTOCOL_ERROR,
          message: "outcome invalid"
        }
      }
    }
    return this.handle(snapshot, context, /* deferred */ null);
  }
  snapshot.stage = {
    id: stage
  }
  return this.resumeSnapshot(snapshot, context, /* deferred */ null, /* value */ null);
}

ConversationHandler.active = new ConversationHandler();
